<?php

/**
 * Broadcast Cron Job
 * This script processes pending broadcasts from the queue
 * Run this script every 1-5 minutes via cron
 */

require_once 'config.php';
require_once 'database.php';
require_once 'functions.php';

// Configuration
define('BATCH_SIZE', 450);  // Process up to 30 users per run
define('DELAY_BETWEEN_MESSAGES', 15000);  // 35ms delay between messages (to avoid rate limits)
define('MAX_RETRIES', 3);  // Retry failed messages up to 3 times

// Log file for cron output
$log_file = __DIR__ . '/broadcast_cron.log';

// Function to log messages
function cron_log(string $message): void {
    global $log_file;
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($log_file, "[$timestamp] $message\n", FILE_APPEND);
    echo "[$timestamp] $message\n";
}

cron_log("=== Broadcast Cron Started ===");

// Get pending broadcasts
$pending_broadcasts = get_pending_broadcasts();

if (empty($pending_broadcasts)) {
    cron_log("No pending broadcasts found.");
    exit(0);
}

cron_log("Found " . count($pending_broadcasts) . " pending broadcast(s).");

// Process each broadcast (usually just one at a time)
foreach ($pending_broadcasts as $broadcast) {
    $broadcast_id = $broadcast['id'];
    $message_data = $broadcast['message_data'];
    $broadcast_type = $broadcast['broadcast_type'];
    $sent_count = $broadcast['sent_count'];
    $failed_count = $broadcast['failed_count'];

    cron_log("Processing broadcast #{$broadcast_id} (Type: {$broadcast_type})");

    // Double-check if broadcast was cancelled (in case it was cancelled after fetching)
    $broadcast_status = get_broadcast_status($broadcast_id);
    if ($broadcast_status && $broadcast_status['status'] === 'cancelled') {
        cron_log("Broadcast #{$broadcast_id} was cancelled. Skipping.");
        continue;
    }

    // Mark as processing if it's the first time
    if ($sent_count == 0 && $failed_count == 0) {
        start_broadcast($broadcast_id);
        cron_log("Broadcast #{$broadcast_id} marked as processing.");
    }
    
    // Get all users
    $all_users = get_all_users();
    $total_users = count($all_users);
    
    cron_log("Total users to process: {$total_users}");
    
    // Calculate which users to process in this batch
    $already_processed = $sent_count + $failed_count;
    $remaining_users = array_slice($all_users, $already_processed, BATCH_SIZE);
    
    if (empty($remaining_users)) {
        // All users have been processed
        complete_broadcast($broadcast_id, $sent_count, $failed_count);
        cron_log("Broadcast #{$broadcast_id} completed. Sent: {$sent_count}, Failed: {$failed_count}");
        continue;
    }
    
    cron_log("Processing batch: " . count($remaining_users) . " users (from {$already_processed} to " . ($already_processed + count($remaining_users)) . ")");
    
    // Process each user in the batch
    foreach ($remaining_users as $user_id) {
        // Check if broadcast was cancelled during processing
        $broadcast_status = get_broadcast_status($broadcast_id);
        if ($broadcast_status && $broadcast_status['status'] === 'cancelled') {
            cron_log("Broadcast #{$broadcast_id} was cancelled during processing. Stopping.");
            update_broadcast_progress($broadcast_id, $sent_count, $failed_count);
            break 2; // Break out of both foreach loops
        }

        try {
            if ($broadcast_type === 'copy') {
                $result = copy_message(
                    $user_id,
                    $message_data['chat_id'],
                    $message_data['message_id']
                );
            } else { // forward
                forward_message(
                    $user_id,
                    $message_data['chat_id'],
                    $message_data['message_id']
                );
                $result = true;
            }

            if ($result !== false) {
                $sent_count++;
            } else {
                $failed_count++;
            }
        } catch (Exception $e) {
            cron_log("Error sending to user {$user_id}: " . $e->getMessage());
            $failed_count++;
        }

        // Update progress every 10 messages
        if (($sent_count + $failed_count) % 10 == 0) {
            update_broadcast_progress($broadcast_id, $sent_count, $failed_count);
        }

        // Sleep to avoid hitting rate limits (30 messages per second max for Telegram)
        usleep(DELAY_BETWEEN_MESSAGES);
    }
    
    // Final update for this batch
    update_broadcast_progress($broadcast_id, $sent_count, $failed_count);
    cron_log("Batch processed. Total sent: {$sent_count}, Failed: {$failed_count}");
    
    // Check if broadcast is complete
    if (($sent_count + $failed_count) >= $total_users) {
        complete_broadcast($broadcast_id, $sent_count, $failed_count);
        cron_log("Broadcast #{$broadcast_id} completed. Sent: {$sent_count}, Failed: {$failed_count}");
        
        // Notify admin about completion (optional)
        notify_admin_broadcast_complete($broadcast_id, $sent_count, $failed_count);
    } else {
        cron_log("Broadcast #{$broadcast_id} will continue in next cron run.");
    }
    
    // Only process one broadcast per cron run to avoid timeouts
    break;
}

// Clean old broadcasts (older than 30 days)
$cleaned = clean_old_broadcasts(30);
if ($cleaned > 0) {
    cron_log("Cleaned {$cleaned} old broadcast(s).");
}

cron_log("=== Broadcast Cron Completed ===\n");

/**
 * Notify admin when broadcast is complete
 */
function notify_admin_broadcast_complete(int $broadcast_id, int $sent, int $failed): void {
    $admin_ids = ADMIN_IDS;
    
    foreach ($admin_ids as $admin_id) {
        try {
            $message = "✅ پخش همگانی #{$broadcast_id} به پایان رسید.\n\n" .
                      "✓ ارسال موفق: {$sent}\n" .
                      "✗ ارسال ناموفق: {$failed}\n" .
                      "📊 مجموع: " . ($sent + $failed);
            
            send_message($admin_id, $message);
        } catch (Exception $e) {
            cron_log("Error notifying admin {$admin_id}: " . $e->getMessage());
        }
    }
}

?>

