<?php

require_once 'config.php';

/**
 * Initialize the database and create tables
 */
function init_db(): void {
    $db = new PDO('sqlite:' . DATABASE);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Create files table
    $db->exec("
        CREATE TABLE IF NOT EXISTS files (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            link_code TEXT UNIQUE,
            file_data TEXT,
            view_count INTEGER DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");
    
    // Create users table
    $db->exec("
        CREATE TABLE IF NOT EXISTS users (
            user_id INTEGER PRIMARY KEY,
            join_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");
    
    // Create channels table
    $db->exec("
        CREATE TABLE IF NOT EXISTS channels (
            chat_id INTEGER PRIMARY KEY,
            channel_link TEXT,
            join_limit INTEGER DEFAULT 0,
            time_limit_hours INTEGER DEFAULT 0,
            is_active INTEGER DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");

    // Create dynamic admins table
    $db->exec("
        CREATE TABLE IF NOT EXISTS dynamic_admins (
            user_id INTEGER PRIMARY KEY,
            added_by INTEGER,
            added_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");

    // Create channel lock stats table
    $db->exec("
        CREATE TABLE IF NOT EXISTS channel_lock_stats (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            channel_chat_id INTEGER,
            action TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(user_id, channel_chat_id, action)
        )
    ");

    // Create fake channels table
    $db->exec("
        CREATE TABLE IF NOT EXISTS fake_channels (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            title TEXT,
            link TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");

    // Create user sessions table
    $db->exec("
        CREATE TABLE IF NOT EXISTS user_sessions (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            link_code TEXT,
            started_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(user_id, link_code)
        )
    ");

    // Create seen/reaction locks table
    $db->exec("
        CREATE TABLE IF NOT EXISTS seen_reaction_locks (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            lock_type TEXT,
            message_text TEXT,
            timer_seconds INTEGER,
            lock_order INTEGER,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");

    // Create user lock progress table
    $db->exec("
        CREATE TABLE IF NOT EXISTS user_lock_progress (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            link_code TEXT,
            lock_id INTEGER,
            started_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            completed_at TIMESTAMP,
            UNIQUE(user_id, link_code, lock_id)
        )
    ");

    // Create likes table
    $db->exec("
        CREATE TABLE IF NOT EXISTS likes (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            link_code TEXT,
            user_id INTEGER,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(link_code, user_id)
        )
    ");
    
    // Create broadcast queue table
    $db->exec("
        CREATE TABLE IF NOT EXISTS broadcast_queue (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            message_data TEXT,
            broadcast_type TEXT,
            status TEXT DEFAULT 'pending',
            total_users INTEGER DEFAULT 0,
            sent_count INTEGER DEFAULT 0,
            failed_count INTEGER DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            started_at TIMESTAMP,
            completed_at TIMESTAMP
        )
    ");

    // Create subscription plans table
    $db->exec("
        CREATE TABLE IF NOT EXISTS subscription_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT,
            price REAL,
            duration_days INTEGER,
            description TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");

    // Create user subscriptions table
    $db->exec("
        CREATE TABLE IF NOT EXISTS user_subscriptions (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            plan_id INTEGER,
            start_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            end_date TIMESTAMP,
            status TEXT DEFAULT 'active',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");

    // Create payment requests table
    $db->exec("
        CREATE TABLE IF NOT EXISTS payment_requests (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            plan_id INTEGER,
            receipt_message_id INTEGER,
            status TEXT DEFAULT 'pending', -- pending, approved, rejected
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");

    // Create bot settings table (key-value store)
    $db->exec("
        CREATE TABLE IF NOT EXISTS bot_settings (
            setting_key TEXT PRIMARY KEY,
            setting_value TEXT
        )
    ");
}

/**
 * Get database connection
 */
function get_db(): PDO {
    $db = new PDO('sqlite:' . DATABASE);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    return $db;
}

/**
 * Migrate existing channels table to add created_at column
 */
function migrate_channels_table(): void {
    $db = get_db();

    // Check if created_at column exists
    $result = $db->query("PRAGMA table_info(channels)");
    $columns = $result->fetchAll(PDO::FETCH_ASSOC);

    $has_created_at = false;
    foreach ($columns as $column) {
        if ($column['name'] === 'created_at') {
            $has_created_at = true;
            break;
        }
    }

    // Add created_at column if it doesn't exist
    if (!$has_created_at) {
        // SQLite doesn't support DEFAULT CURRENT_TIMESTAMP in ALTER TABLE
        // Use a static default instead
        $db->exec("ALTER TABLE channels ADD COLUMN created_at TIMESTAMP DEFAULT '2024-01-01 00:00:00'");

        // Update existing rows to have current timestamp
        $db->exec("UPDATE channels SET created_at = datetime('now') WHERE created_at = '2024-01-01 00:00:00'");
    }
}

/**
 * Add a file record
 */
function add_file_record(string $link_code, array $file_data): void {
    $db = get_db();
    $stmt = $db->prepare("INSERT INTO files (link_code, file_data) VALUES (?, ?)");
    $stmt->execute([$link_code, json_encode($file_data)]);
}

/**
 * Get file record by link code
 */
function get_file_record(string $link_code): ?array {
    $db = get_db();
    $stmt = $db->prepare("SELECT id, file_data, view_count FROM files WHERE link_code = ?");
    $stmt->execute([$link_code]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($row) {
        return [
            'id' => $row['id'],
            'file_data' => json_decode($row['file_data'], true),
            'view_count' => $row['view_count']
        ];
    }
    return null;
}

/**
 * Increment view count for a file
 */
function increment_view_count(string $link_code): void {
    $db = get_db();
    $stmt = $db->prepare("UPDATE files SET view_count = view_count + 1 WHERE link_code = ?");
    $stmt->execute([$link_code]);
}

/**
 * Add a user to the database
 */
function add_user(int $user_id): void {
    $db = get_db();
    $stmt = $db->prepare("INSERT OR IGNORE INTO users (user_id) VALUES (?)");
    $stmt->execute([$user_id]);
}

/**
 * Get all users
 */
function get_all_users(): array {
    $db = get_db();
    $stmt = $db->query("SELECT user_id FROM users");
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

/**
 * Get user count
 */
function get_user_count(): int {
    $db = get_db();
    $stmt = $db->query("SELECT COUNT(*) FROM users");
    return (int)$stmt->fetchColumn();
}

/**
 * Add a channel
 */
function add_channel(int $chat_id, string $channel_link): void {
    $db = get_db();
    $stmt = $db->prepare("INSERT OR REPLACE INTO channels (chat_id, channel_link) VALUES (?, ?)");
    $stmt->execute([$chat_id, $channel_link]);
}

/**
 * Remove a channel
 */
function remove_channel(int $chat_id): void {
    $db = get_db();
    $stmt = $db->prepare("DELETE FROM channels WHERE chat_id = ?");
    $stmt->execute([$chat_id]);
}

/**
 * Get all channels (only active ones by default)
 */
function get_channels(bool $only_active = true): array {
    $db = get_db();

    if ($only_active) {
        $stmt = $db->query("SELECT chat_id, channel_link, join_limit, time_limit_hours, is_active, created_at FROM channels WHERE is_active = 1");
    } else {
        $stmt = $db->query("SELECT chat_id, channel_link, join_limit, time_limit_hours, is_active, created_at FROM channels");
    }

    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $channels = [];
    foreach ($rows as $row) {
        // Double-check active status based on limits
        $is_truly_active = $only_active ? is_channel_active($row['chat_id']) : true;

        if (!$only_active || $is_truly_active) {
            $channels[] = [
                'chat_id' => $row['chat_id'],
                'channel_link' => $row['channel_link'],
                'join_limit' => $row['join_limit'] ?? 0,
                'time_limit_hours' => $row['time_limit_hours'] ?? 0,
                'is_active' => $row['is_active'] ?? 1,
                'created_at' => $row['created_at'] ?? null
            ];
        }
    }
    return $channels;
}

/**
 * Add a like
 */
function add_like(string $link_code, int $user_id): bool {
    $db = get_db();
    try {
        $stmt = $db->prepare("INSERT INTO likes (link_code, user_id) VALUES (?, ?)");
        $stmt->execute([$link_code, $user_id]);
        return true;
    } catch (PDOException $e) {
        // User already liked this link (unique constraint violation)
        return false;
    }
}

/**
 * Remove a like
 */
function remove_like(string $link_code, int $user_id): bool {
    $db = get_db();
    $stmt = $db->prepare("DELETE FROM likes WHERE link_code = ? AND user_id = ?");
    $stmt->execute([$link_code, $user_id]);
    return $stmt->rowCount() > 0;
}

/**
 * Get like count for a link
 */
function get_like_count(string $link_code): int {
    $db = get_db();
    $stmt = $db->prepare("SELECT COUNT(*) FROM likes WHERE link_code = ?");
    $stmt->execute([$link_code]);
    return (int)$stmt->fetchColumn();
}

/**
 * Check if user has liked a link
 */
function has_user_liked(string $link_code, int $user_id): bool {
    $db = get_db();
    $stmt = $db->prepare("SELECT 1 FROM likes WHERE link_code = ? AND user_id = ?");
    $stmt->execute([$link_code, $user_id]);
    return $stmt->fetch() !== false;
}

/**
 * Add a broadcast to the queue
 */
function queue_broadcast(array $message_data, string $broadcast_type): int {
    $db = get_db();
    $total_users = get_user_count();
    
    $stmt = $db->prepare("
        INSERT INTO broadcast_queue (message_data, broadcast_type, total_users) 
        VALUES (?, ?, ?)
    ");
    $stmt->execute([json_encode($message_data), $broadcast_type, $total_users]);
    
    return (int)$db->lastInsertId();
}

/**
 * Get pending broadcasts
 */
function get_pending_broadcasts(): array {
    $db = get_db();
    $stmt = $db->query("
        SELECT id, message_data, broadcast_type, total_users, sent_count, failed_count
        FROM broadcast_queue
        WHERE status IN ('pending', 'processing')
        ORDER BY created_at ASC
    ");
    
    $broadcasts = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $broadcasts[] = [
            'id' => $row['id'],
            'message_data' => json_decode($row['message_data'], true),
            'broadcast_type' => $row['broadcast_type'],
            'total_users' => $row['total_users'],
            'sent_count' => $row['sent_count'],
            'failed_count' => $row['failed_count']
        ];
    }
    
    return $broadcasts;
}

/**
 * Update broadcast status to processing
 */
function start_broadcast(int $broadcast_id): void {
    $db = get_db();
    $stmt = $db->prepare("
        UPDATE broadcast_queue 
        SET status = 'processing', started_at = CURRENT_TIMESTAMP 
        WHERE id = ?
    ");
    $stmt->execute([$broadcast_id]);
}

/**
 * Update broadcast progress
 */
function update_broadcast_progress(int $broadcast_id, int $sent_count, int $failed_count): void {
    $db = get_db();
    $stmt = $db->prepare("
        UPDATE broadcast_queue 
        SET sent_count = ?, failed_count = ? 
        WHERE id = ?
    ");
    $stmt->execute([$sent_count, $failed_count, $broadcast_id]);
}

/**
 * Complete a broadcast
 */
function complete_broadcast(int $broadcast_id, int $sent_count, int $failed_count): void {
    $db = get_db();
    $stmt = $db->prepare("
        UPDATE broadcast_queue 
        SET status = 'completed', sent_count = ?, failed_count = ?, completed_at = CURRENT_TIMESTAMP 
        WHERE id = ?
    ");
    $stmt->execute([$sent_count, $failed_count, $broadcast_id]);
}

/**
 * Get broadcast status
 */
function get_broadcast_status(int $broadcast_id): ?array {
    $db = get_db();
    $stmt = $db->prepare("
        SELECT id, status, total_users, sent_count, failed_count, created_at, started_at, completed_at 
        FROM broadcast_queue 
        WHERE id = ?
    ");
    $stmt->execute([$broadcast_id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    
    return $row ?: null;
}

/**
 * Get all broadcasts (for admin stats)
 */
function get_all_broadcasts(int $limit = 10): array {
    $db = get_db();
    $stmt = $db->prepare("
        SELECT id, broadcast_type, status, total_users, sent_count, failed_count, created_at, completed_at 
        FROM broadcast_queue 
        ORDER BY created_at DESC 
        LIMIT ?
    ");
    $stmt->execute([$limit]);
    
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Clean old completed broadcasts
 */
function clean_old_broadcasts(int $days = 30): int {
    $db = get_db();
    $stmt = $db->prepare("
        DELETE FROM broadcast_queue
        WHERE status = 'completed'
        AND completed_at < datetime('now', '-' || ? || ' days')
    ");
    $stmt->execute([$days]);

    return $stmt->rowCount();
}

/**
 * Cancel a broadcast (pending or processing)
 */
function cancel_broadcast(int $broadcast_id): bool {
    $db = get_db();
    $stmt = $db->prepare("
        UPDATE broadcast_queue
        SET status = 'cancelled', completed_at = CURRENT_TIMESTAMP
        WHERE id = ? AND status IN ('pending', 'processing')
    ");
    $stmt->execute([$broadcast_id]);

    return $stmt->rowCount() > 0;
}

/**
 * Get detailed broadcast information
 */
function get_broadcast_details(int $broadcast_id): ?array {
    $db = get_db();
    $stmt = $db->prepare("
        SELECT id, message_data, broadcast_type, status, total_users,
               sent_count, failed_count, created_at, started_at, completed_at
        FROM broadcast_queue
        WHERE id = ?
    ");
    $stmt->execute([$broadcast_id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($row) {
        $row['message_data'] = json_decode($row['message_data'], true);
        return $row;
    }

    return null;
}

/**
 * Get recent broadcasts with pagination
 */
function get_recent_broadcasts(int $limit = 10, int $offset = 0): array {
    $db = get_db();
    $stmt = $db->prepare("
        SELECT id, broadcast_type, status, total_users, sent_count, failed_count,
               created_at, started_at, completed_at
        FROM broadcast_queue
        ORDER BY created_at DESC
        LIMIT ? OFFSET ?
    ");
    $stmt->execute([$limit, $offset]);

    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Track user interaction with channel lock
 * Actions: 'viewed', 'already_member', 'joined', 'not_joined'
 */
function track_channel_interaction(int $user_id, int $channel_chat_id, string $action): void {
    $db = get_db();
    try {
        $stmt = $db->prepare("
            INSERT OR IGNORE INTO channel_lock_stats (user_id, channel_chat_id, action)
            VALUES (?, ?, ?)
        ");
        $stmt->execute([$user_id, $channel_chat_id, $action]);
    } catch (PDOException $e) {
        // Silently fail if there's a duplicate entry
        error_log("Channel tracking error: " . $e->getMessage());
    }
}

/**
 * Get channel lock statistics
 */
function get_channel_statistics(int $channel_chat_id): array {
    $db = get_db();

    // Get total users who viewed the lock (started bot and saw this channel requirement)
    $stmt = $db->prepare("
        SELECT COUNT(DISTINCT user_id) as count
        FROM channel_lock_stats
        WHERE channel_chat_id = ? AND action = 'viewed'
    ");
    $stmt->execute([$channel_chat_id]);
    $total_viewed = (int)$stmt->fetchColumn();

    // Get users who were already members
    $stmt = $db->prepare("
        SELECT COUNT(DISTINCT user_id) as count
        FROM channel_lock_stats
        WHERE channel_chat_id = ? AND action = 'already_member'
    ");
    $stmt->execute([$channel_chat_id]);
    $already_members = (int)$stmt->fetchColumn();

    // Get users who joined through the lock
    $stmt = $db->prepare("
        SELECT COUNT(DISTINCT user_id) as count
        FROM channel_lock_stats
        WHERE channel_chat_id = ? AND action = 'joined'
    ");
    $stmt->execute([$channel_chat_id]);
    $joined = (int)$stmt->fetchColumn();

    // Calculate users who didn't join (viewed but not already member and not joined)
    $not_joined = $total_viewed - $already_members - $joined;

    // Get channel info
    $stmt = $db->prepare("SELECT channel_link, created_at FROM channels WHERE chat_id = ?");
    $stmt->execute([$channel_chat_id]);
    $channel_info = $stmt->fetch(PDO::FETCH_ASSOC);

    return [
        'channel_link' => $channel_info['channel_link'] ?? '',
        'created_at' => $channel_info['created_at'] ?? null,
        'total_viewed' => $total_viewed,
        'already_members' => $already_members,
        'joined' => $joined,
        'not_joined' => max(0, $not_joined)
    ];
}

/**
 * Get channel by chat_id
 */
function get_channel_by_id(int $chat_id): ?array {
    $db = get_db();
    $stmt = $db->prepare("SELECT chat_id, channel_link, join_limit, time_limit_hours, is_active, created_at FROM channels WHERE chat_id = ?");
    $stmt->execute([$chat_id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    return $row ?: null;
}

/**
 * Add a fake channel
 */
function add_fake_channel(string $title, string $link): int {
    $db = get_db();
    $stmt = $db->prepare("INSERT INTO fake_channels (title, link) VALUES (?, ?)");
    $stmt->execute([$title, $link]);

    return (int)$db->lastInsertId();
}

/**
 * Remove a fake channel
 */
function remove_fake_channel(int $id): void {
    $db = get_db();
    $stmt = $db->prepare("DELETE FROM fake_channels WHERE id = ?");
    $stmt->execute([$id]);
}

/**
 * Get all fake channels
 */
function get_fake_channels(): array {
    $db = get_db();
    $stmt = $db->query("SELECT id, title, link, created_at FROM fake_channels ORDER BY created_at DESC");

    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get fake channel by id
 */
function get_fake_channel_by_id(int $id): ?array {
    $db = get_db();
    $stmt = $db->prepare("SELECT id, title, link, created_at FROM fake_channels WHERE id = ?");
    $stmt->execute([$id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    return $row ?: null;
}

/**
 * Start user session for file access
 */
function start_user_session(int $user_id, string $link_code): void {
    $db = get_db();
    try {
        $stmt = $db->prepare("
            INSERT OR REPLACE INTO user_sessions (user_id, link_code, started_at)
            VALUES (?, ?, CURRENT_TIMESTAMP)
        ");
        $stmt->execute([$user_id, $link_code]);
    } catch (PDOException $e) {
        error_log("Session tracking error: " . $e->getMessage());
    }
}

/**
 * Get session elapsed time in seconds
 */
function get_session_elapsed_time(int $user_id, string $link_code): int {
    $db = get_db();
    $stmt = $db->prepare("
        SELECT
            CAST((julianday('now') - julianday(started_at)) * 86400 AS INTEGER) as elapsed
        FROM user_sessions
        WHERE user_id = ? AND link_code = ?
    ");
    $stmt->execute([$user_id, $link_code]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);

    return $result ? (int)$result['elapsed'] : 0;
}

/**
 * Clear user session after successful access
 */
function clear_user_session(int $user_id, string $link_code): void {
    $db = get_db();
    $stmt = $db->prepare("DELETE FROM user_sessions WHERE user_id = ? AND link_code = ?");
    $stmt->execute([$user_id, $link_code]);
}

/**
 * Clean old sessions (older than 24 hours)
 */
function clean_old_sessions(): int {
    $db = get_db();
    $stmt = $db->prepare("
        DELETE FROM user_sessions
        WHERE started_at < datetime('now', '-1 day')
    ");
    $stmt->execute();

    return $stmt->rowCount();
}

/**
 * Add a seen/reaction lock
 */
function add_seen_reaction_lock(string $lock_type, string $message_text, int $timer_seconds): int {
    $db = get_db();

    // Get current max order
    $stmt = $db->query("SELECT COALESCE(MAX(lock_order), 0) as max_order FROM seen_reaction_locks");
    $max_order = (int)$stmt->fetchColumn();

    $stmt = $db->prepare("
        INSERT INTO seen_reaction_locks (lock_type, message_text, timer_seconds, lock_order)
        VALUES (?, ?, ?, ?)
    ");
    $stmt->execute([$lock_type, $message_text, $timer_seconds, $max_order + 1]);

    return (int)$db->lastInsertId();
}

/**
 * Remove a seen/reaction lock
 */
function remove_seen_reaction_lock(int $id): void {
    $db = get_db();

    // Get the order of the lock being deleted
    $stmt = $db->prepare("SELECT lock_order FROM seen_reaction_locks WHERE id = ?");
    $stmt->execute([$id]);
    $deleted_order = $stmt->fetchColumn();

    // Delete the lock
    $stmt = $db->prepare("DELETE FROM seen_reaction_locks WHERE id = ?");
    $stmt->execute([$id]);

    // Reorder remaining locks
    if ($deleted_order !== false) {
        $stmt = $db->prepare("
            UPDATE seen_reaction_locks
            SET lock_order = lock_order - 1
            WHERE lock_order > ?
        ");
        $stmt->execute([$deleted_order]);
    }
}

/**
 * Get all seen/reaction locks (ordered)
 */
function get_seen_reaction_locks(): array {
    $db = get_db();
    $stmt = $db->query("
        SELECT id, lock_type, message_text, timer_seconds, lock_order, created_at
        FROM seen_reaction_locks
        ORDER BY lock_order ASC
    ");

    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get seen/reaction lock by ID
 */
function get_seen_reaction_lock_by_id(int $id): ?array {
    $db = get_db();
    $stmt = $db->prepare("
        SELECT id, lock_type, message_text, timer_seconds, lock_order, created_at
        FROM seen_reaction_locks
        WHERE id = ?
    ");
    $stmt->execute([$id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    return $row ?: null;
}

/**
 * Get lock count
 */
function get_seen_reaction_lock_count(): int {
    $db = get_db();
    $stmt = $db->query("SELECT COUNT(*) FROM seen_reaction_locks");
    return (int)$stmt->fetchColumn();
}

/**
 * Start user lock progress
 */
function start_user_lock(int $user_id, string $link_code, int $lock_id): void {
    $db = get_db();
    try {
        $stmt = $db->prepare("
            INSERT OR REPLACE INTO user_lock_progress (user_id, link_code, lock_id, started_at)
            VALUES (?, ?, ?, CURRENT_TIMESTAMP)
        ");
        $stmt->execute([$user_id, $link_code, $lock_id]);
    } catch (PDOException $e) {
        error_log("Lock progress error: " . $e->getMessage());
    }
}

/**
 * Complete user lock
 */
function complete_user_lock(int $user_id, string $link_code, int $lock_id): void {
    $db = get_db();
    $stmt = $db->prepare("
        UPDATE user_lock_progress
        SET completed_at = CURRENT_TIMESTAMP
        WHERE user_id = ? AND link_code = ? AND lock_id = ?
    ");
    $stmt->execute([$user_id, $link_code, $lock_id]);
}

/**
 * Get user lock elapsed time in seconds
 */
function get_user_lock_elapsed_time(int $user_id, string $link_code, int $lock_id): int {
    $db = get_db();
    $stmt = $db->prepare("
        SELECT
            CAST((julianday('now') - julianday(started_at)) * 86400 AS INTEGER) as elapsed
        FROM user_lock_progress
        WHERE user_id = ? AND link_code = ? AND lock_id = ?
    ");
    $stmt->execute([$user_id, $link_code, $lock_id]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);

    return $result ? (int)$result['elapsed'] : 0;
}

/**
 * Check if user completed a lock
 */
function is_user_lock_completed(int $user_id, string $link_code, int $lock_id): bool {
    $db = get_db();
    $stmt = $db->prepare("
        SELECT completed_at FROM user_lock_progress
        WHERE user_id = ? AND link_code = ? AND lock_id = ?
    ");
    $stmt->execute([$user_id, $link_code, $lock_id]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);

    return $result && $result['completed_at'] !== null;
}

/**
 * Get next incomplete lock for user
 */
function get_next_user_lock(int $user_id, string $link_code): ?array {
    $db = get_db();

    // Get all locks ordered
    $all_locks = get_seen_reaction_locks();

    foreach ($all_locks as $lock) {
        if (!is_user_lock_completed($user_id, $link_code, $lock['id'])) {
            return $lock;
        }
    }

    return null; // All locks completed
}

/**
 * Clear all user lock progress for a link
 */
function clear_user_lock_progress(int $user_id, string $link_code): void {
    $db = get_db();
    $stmt = $db->prepare("DELETE FROM user_lock_progress WHERE user_id = ? AND link_code = ?");
    $stmt->execute([$user_id, $link_code]);
}

/**
 * Get all file links with pagination
 */
function get_all_file_links(int $limit = 50, int $offset = 0): array {
    $db = get_db();
    $stmt = $db->prepare("
        SELECT link_code, view_count, created_at
        FROM files
        ORDER BY created_at DESC
        LIMIT ? OFFSET ?
    ");
    $stmt->execute([$limit, $offset]);

    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get total count of file links
 */
function get_file_links_count(): int {
    $db = get_db();
    $stmt = $db->query("SELECT COUNT(*) FROM files");
    return (int)$stmt->fetchColumn();
}

/**
 * Update file data for a link
 */
function update_file_data(string $link_code, array $new_file_data): void {
    $db = get_db();
    $stmt = $db->prepare("UPDATE files SET file_data = ? WHERE link_code = ?");
    $stmt->execute([json_encode($new_file_data), $link_code]);
}

/**
 * Delete a file link completely
 */
function delete_file_link(string $link_code): void {
    $db = get_db();
    $stmt = $db->prepare("DELETE FROM files WHERE link_code = ?");
    $stmt->execute([$link_code]);
}

/**
 * Update channel limits
 */
function update_channel_limits(int $chat_id, int $join_limit, int $time_limit_hours): void {
    $db = get_db();
    $stmt = $db->prepare("UPDATE channels SET join_limit = ?, time_limit_hours = ? WHERE chat_id = ?");
    $stmt->execute([$join_limit, $time_limit_hours, $chat_id]);
}

/**
 * Check if channel is active based on limits
 */
function is_channel_active(int $chat_id): bool {
    $db = get_db();
    $channel = get_channel_by_id($chat_id);

    if (!$channel) {
        return false;
    }

    // Check if manually disabled
    if (isset($channel['is_active']) && $channel['is_active'] == 0) {
        return false;
    }

    // Check time limit
    if ($channel['time_limit_hours'] > 0) {
        $created_time = strtotime($channel['created_at']);
        $expire_time = $created_time + ($channel['time_limit_hours'] * 3600);

        if (time() > $expire_time) {
            // Auto-disable channel
            disable_channel($chat_id);
            return false;
        }
    }

    // Check join limit
    if ($channel['join_limit'] > 0) {
        $stats = get_channel_statistics($chat_id);
        if ($stats['joined'] >= $channel['join_limit']) {
            // Auto-disable channel
            disable_channel($chat_id);
            return false;
        }
    }

    return true;
}

/**
 * Disable a channel
 */
function disable_channel(int $chat_id): void {
    $db = get_db();
    $stmt = $db->prepare("UPDATE channels SET is_active = 0 WHERE chat_id = ?");
    $stmt->execute([$chat_id]);
}

/**
 * Enable a channel
 */
function enable_channel(int $chat_id): void {
    $db = get_db();
    $stmt = $db->prepare("UPDATE channels SET is_active = 1 WHERE chat_id = ?");
    $stmt->execute([$chat_id]);
}

/**
 * Get all admins (config + dynamic)
 */
function get_all_admins(): array {
    $config_admins = ADMIN_IDS;

    $db = get_db();
    $stmt = $db->query("SELECT user_id FROM dynamic_admins");
    $dynamic_admins = $stmt->fetchAll(PDO::FETCH_COLUMN);

    return array_unique(array_merge($config_admins, $dynamic_admins));
}

/**
 * Add a dynamic admin
 */
function add_dynamic_admin(int $user_id, int $added_by): bool {
    $db = get_db();
    try {
        $stmt = $db->prepare("INSERT INTO dynamic_admins (user_id, added_by) VALUES (?, ?)");
        $stmt->execute([$user_id, $added_by]);
        return true;
    } catch (PDOException $e) {
        return false;
    }
}

/**
 * Remove a dynamic admin
 */
function remove_dynamic_admin(int $user_id): bool {
    // Don't allow removing config admins
    if (in_array($user_id, ADMIN_IDS)) {
        return false;
    }

    try {
        $db = get_db();
        $stmt = $db->prepare("DELETE FROM dynamic_admins WHERE user_id = ?");
        $stmt->execute([$user_id]);
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        error_log("Error removing dynamic admin: " . $e->getMessage());
        return false;
    }
}

/**
 * Check if user is a config admin (immutable)
 */
function is_config_admin(int $user_id): bool {
    return in_array($user_id, ADMIN_IDS);
}

/**
 * Get all dynamic admins with details
 */
function get_dynamic_admins(): array {
    try {
        $db = get_db();
        $stmt = $db->query("SELECT user_id, added_by, added_at FROM dynamic_admins ORDER BY added_at DESC");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // Table might not exist - run migration first
        error_log("Error getting dynamic admins: " . $e->getMessage());
        return [];
    }
}

?>


