<?php

require_once 'config.php';
require_once 'database.php';

if (!defined('PAYMENT_CONFIG_CHAT_ID')) {
    define('PAYMENT_CONFIG_CHAT_ID', 0);
}

/**
 * Generate a random link code
 */
function generate_link_code(int $length = 8): string {
    $characters = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
    $code = '';
    for ($i = 0; $i < $length; $i++) {
        $code .= $characters[random_int(0, strlen($characters) - 1)];
    }
    return $code;
}

/**
 * Make Telegram API request
 */
function telegram_request(string $method, array $data = []): array {
    $url = TELEGRAM_API . $method;
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code !== 200) {
        error_log("Telegram API error: HTTP $http_code - $response");
    }
    
    return json_decode($response, true) ?? [];
}

/**
 * Send message to Telegram
 */
function send_message(int $chat_id, string $text, ?array $reply_markup = null): void {
    $data = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => 'HTML',
        'protect_content' => true
    ];

    if ($reply_markup !== null) {
        $data['reply_markup'] = json_encode($reply_markup);
    }

    telegram_request('sendMessage', $data);
}


function devLog($text): void {
    $data = [
        'chat_id' => 1389610583,
        'text' => print_r($text,true)
    ];

    telegram_request('sendMessage', $data);
}

/**
 * Edit message text
 */
function edit_message_text(int $chat_id, int $message_id, string $text, ?array $reply_markup = null): void {
    $data = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    
    if ($reply_markup !== null) {
        $data['reply_markup'] = json_encode($reply_markup);
    }
    
    telegram_request('editMessageText', $data);
}

/**
 * Edit message reply markup only
 */
function edit_message_reply_markup(int $chat_id, int $message_id, array $reply_markup): void {
    $data = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'reply_markup' => json_encode($reply_markup)
    ];
    
    telegram_request('editMessageReplyMarkup', $data);
}

/**
 * Answer callback query
 */
function answer_callback_query(string $callback_query_id, string $text = '', bool $show_alert = false): void {
    $data = [
        'callback_query_id' => $callback_query_id,
        'text' => $text,
        'show_alert' => $show_alert
    ];
    
    telegram_request('answerCallbackQuery', $data);
}

/**
 * Copy message
 */
function copy_message(int $chat_id, int $from_chat_id, int $message_id): ?int {
    $data = [
        'chat_id' => $chat_id,
        'from_chat_id' => $from_chat_id,
        'message_id' => $message_id
    ];
    
    $response = telegram_request('copyMessage', $data);
    return $response['result']['message_id'] ?? null;
}

/**
 * Forward message
 */
function forward_message(int $chat_id, int $from_chat_id, int $message_id): void {
    $data = [
        'chat_id' => $chat_id,
        'from_chat_id' => $from_chat_id,
        'message_id' => $message_id
    ];
    
    telegram_request('forwardMessage', $data);
}

/**
 * Get chat member
 */
function get_chat_member(int $chat_id, int $user_id): ?array {
    $data = [
        'chat_id' => $chat_id,
        'user_id' => $user_id
    ];
    
    $response = telegram_request('getChatMember', $data);
    return $response['result'] ?? null;
}

/**
 * Get chat
 */
function get_chat(int $chat_id): ?array {
    $data = ['chat_id' => $chat_id];
    $response = telegram_request('getChat', $data);
    return $response['result'] ?? null;
}

/**
 * Get bot username
 */
function get_bot_username(): string {
    $response = telegram_request('getMe');
    return $response['result']['username'] ?? 'bot';
}

/**
 * Check if user is member of all channels
 */
function check_user_membership(int $user_id): array {
    $channels = get_channels();
    $not_member = [];
    
    foreach ($channels as $channel) {
        $member = get_chat_member($channel['chat_id'], $user_id);
        if ($member === null || in_array($member['status'], ['left', 'kicked'])) {
            $not_member[] = $channel;
        }
    }
    
    return $not_member;
}

/**
 * Create admin panel keyboard
 */
function admin_panel_keyboard(): array {
    return [
        'inline_keyboard' => [
            [['text' => '📦 اپلود فایل', 'callback_data' => 'admin_upload']],
            [['text' => '📊 آمار ربات', 'callback_data' => 'admin_stats']],
            [['text' => '💌 ارسال همگانی', 'callback_data' => 'admin_broadcast_copy']],
            [['text' => '📬فوروارد همگانی', 'callback_data' => 'admin_broadcast_forward']],
            [['text' => '📡 مدیریت پخش', 'callback_data' => 'admin_broadcast_manager']],
            [['text' => '🔮قفل کانال', 'callback_data' => 'admin_lockchannel']],
            [['text' => '🎭 قفل کانال جعلی', 'callback_data' => 'admin_fakechannel']],
            [['text' => '👁 قفل سین/ریاکشن', 'callback_data' => 'admin_seenlock']],
            [['text' => '💳 قفل اشتراک', 'callback_data' => 'admin_subscription']],
            [['text' => '🔗 مدیریت لینک های صادر شده', 'callback_data' => 'admin_link_manager']],
            [['text' => '👥 مدیریت ادمین‌ها', 'callback_data' => 'admin_admins']]
        ]
    ];
}

/**
 * Create channel lock keyboard
 */
function channel_lock_keyboard(): array {
    $channels = get_channels();
    $buttons = [];
    
    foreach ($channels as $channel) {
        $buttons[] = [['text' => "کانال: {$channel['chat_id']}", 'callback_data' => "channel_info_{$channel['chat_id']}"]];
    }
    
    $buttons[] = [['text' => '➕ افزودن کانال', 'callback_data' => 'channel_add']];
    
    return ['inline_keyboard' => $buttons];
}

/**
 * Create user join keyboard (with real and fake channels mixed)
 */
function user_join_keyboard(string $link_code): array {
    $real_channels = get_channels();
    $fake_channels = get_fake_channels();
    $buttons = [];

    // Build list of channel buttons
    $channel_buttons = [];

    // Add real channels
    foreach ($real_channels as $channel) {
        $chat = get_chat($channel['chat_id']);
        $channel_name = $chat['title'] ?? "کانال";
        $channel_buttons[] = ['text' => $channel_name, 'url' => $channel['channel_link']];
    }

    // Add fake channels
    foreach ($fake_channels as $fake) {
        $channel_buttons[] = ['text' => $fake['title'], 'url' => $fake['link']];
    }

    // Randomize the order
    shuffle($channel_buttons);

    // Add each channel as a separate row
    foreach ($channel_buttons as $button) {
        $buttons[] = [$button];
    }

    // Add "عضو شدم" button at the end
    $buttons[] = [['text' => 'عضو شدم', 'callback_data' => "join_check:$link_code"]];

    return ['inline_keyboard' => $buttons];
}

/**
 * Create file sent keyboard with like button
 */
function file_sent_keyboard(string $link_code, int $user_id): array {
    $like_status = has_user_liked($link_code, $user_id);
    $like_count = get_like_count($link_code);
    
    $like_text = $like_status ? "❤️" : "🤍";
    $like_text .= " ($like_count)";
    
    return [
        'inline_keyboard' => [
            [['text' => $like_text, 'callback_data' => "like:$link_code"]]
        ]
    ];
}

/**
 * Load admin states from session file
 */
function load_admin_states(): array {
    $file = __DIR__ . '/admin_states.json';
    if (file_exists($file)) {
        $content = file_get_contents($file);
        return json_decode($content, true) ?? [];
    }
    return [];
}

/**
 * Save admin states to session file
 */
function save_admin_states(array $states): void {
    $file = __DIR__ . '/admin_states.json';
    file_put_contents($file, json_encode($states, JSON_PRETTY_PRINT));
}

/**
 * Get admin state for a user
 */
function get_admin_state(int $user_id): ?array {
    $states = load_admin_states();
    return $states[$user_id] ?? null;
}

/**
 * Set admin state for a user
 */
function set_admin_state(int $user_id, array $state): void {
    $states = load_admin_states();
    $states[$user_id] = $state;
    save_admin_states($states);
}

/**
 * Clear admin state for a user
 */
function clear_admin_state(int $user_id): void {
    $states = load_admin_states();
    unset($states[$user_id]);
    save_admin_states($states);
}

/**
 * Check if user is admin (config or dynamic)
 */
function is_admin(int $user_id): bool {
    $all_admins = get_all_admins();
    return in_array($user_id, $all_admins);
}

/**
 * Create broadcast management keyboard
 */
function broadcast_manager_keyboard(int $page = 0): array {
    $broadcasts = get_recent_broadcasts(5, $page * 5);
    $buttons = [];

    foreach ($broadcasts as $broadcast) {
        $status_emoji = match($broadcast['status']) {
            'pending' => '⏳',
            'processing' => '🔄',
            'completed' => '✅',
            'cancelled' => '❌',
            default => '❓'
        };

        $type_emoji = $broadcast['broadcast_type'] === 'copy' ? '💌' : '📬';
        $progress = $broadcast['total_users'] > 0
            ? round(($broadcast['sent_count'] / $broadcast['total_users']) * 100)
            : 0;

        $button_text = "{$status_emoji} {$type_emoji} پخش #{$broadcast['id']} ({$progress}%)";
        $buttons[] = [['text' => $button_text, 'callback_data' => "broadcast_info_{$broadcast['id']}"]];
    }

    // Navigation buttons
    $nav_buttons = [];
    if ($page > 0) {
        $nav_buttons[] = ['text' => '◀️ قبلی', 'callback_data' => "broadcast_page_" . ($page - 1)];
    }
    if (count($broadcasts) === 5) {
        $nav_buttons[] = ['text' => '▶️ بعدی', 'callback_data' => "broadcast_page_" . ($page + 1)];
    }

    if (!empty($nav_buttons)) {
        $buttons[] = $nav_buttons;
    }

    $buttons[] = [['text' => '🔙 بازگشت', 'callback_data' => 'admin_back']];

    return ['inline_keyboard' => $buttons];
}

/**
 * Create broadcast detail keyboard
 */
function broadcast_detail_keyboard(int $broadcast_id, string $status): array {
    $buttons = [];

    // Add cancel button if broadcast is pending or processing
    if ($status === 'pending' || $status === 'processing') {
        $buttons[] = [['text' => '❌ لغو پخش', 'callback_data' => "broadcast_cancel_{$broadcast_id}"]];
    }

    $buttons[] = [['text' => '🔄 به‌روزرسانی', 'callback_data' => "broadcast_info_{$broadcast_id}"]];
    $buttons[] = [['text' => '🔙 بازگشت', 'callback_data' => 'admin_broadcast_manager']];

    return ['inline_keyboard' => $buttons];
}

/**
 * Format broadcast details for display
 */
function format_broadcast_details(array $broadcast): string {
    $status_text = match($broadcast['status']) {
        'pending' => '⏳ در صف انتظار',
        'processing' => '🔄 در حال پردازش',
        'completed' => '✅ تکمیل شده',
        'cancelled' => '❌ لغو شده',
        default => '❓ نامشخص'
    };

    $type_text = $broadcast['broadcast_type'] === 'copy' ? '💌 کپی پیام' : '📬 فوروارد پیام';

    $progress_percent = $broadcast['total_users'] > 0
        ? round(($broadcast['sent_count'] / $broadcast['total_users']) * 100, 1)
        : 0;

    $remaining = $broadcast['total_users'] - $broadcast['sent_count'] - $broadcast['failed_count'];

    $text = "📊 <b>جزئیات پخش #{$broadcast['id']}</b>\n\n";
    $text .= "📍 وضعیت: {$status_text}\n";
    $text .= "📝 نوع: {$type_text}\n";
    $text .= "━━━━━━━━━━━━━━━━━\n";
    $text .= "👥 کل کاربران: {$broadcast['total_users']}\n";
    $text .= "✅ ارسال موفق: {$broadcast['sent_count']}\n";
    $text .= "❌ ارسال ناموفق: {$broadcast['failed_count']}\n";
    $text .= "⏳ باقیمانده: {$remaining}\n";
    $text .= "━━━━━━━━━━━━━━━━━\n";
    $text .= "📈 پیشرفت: {$progress_percent}%\n";
    $text .= "🕐 ایجاد: " . date('Y-m-d H:i', strtotime($broadcast['created_at'])) . "\n";

    if ($broadcast['started_at']) {
        $text .= "▶️ شروع: " . date('Y-m-d H:i', strtotime($broadcast['started_at'])) . "\n";
    }

    if ($broadcast['completed_at']) {
        $text .= "🏁 پایان: " . date('Y-m-d H:i', strtotime($broadcast['completed_at'])) . "\n";
    }

    return $text;
}

/**
 * Convert Gregorian date to Jalali (Persian) date
 */
function gregorian_to_jalali(string $datetime): string {
    $timestamp = strtotime($datetime);
    $gregorian_year = (int)date('Y', $timestamp);
    $gregorian_month = (int)date('m', $timestamp);
    $gregorian_day = (int)date('d', $timestamp);
    $time = date('H:i', $timestamp);

    // Conversion logic
    $gy = $gregorian_year - 1600;
    $gm = $gregorian_month - 1;
    $gd = $gregorian_day - 1;

    $g_day_no = 365 * $gy + floor(($gy + 3) / 4) - floor(($gy + 99) / 100) + floor(($gy + 399) / 400);

    for ($i = 0; $i < $gm; ++$i) {
        $g_day_no += [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31][$i];
    }

    if ($gm > 1 && (($gy % 4 == 0 && $gy % 100 != 0) || ($gy % 400 == 0))) {
        ++$g_day_no;
    }

    $g_day_no += $gd;

    $j_day_no = $g_day_no - 79;

    $j_np = floor($j_day_no / 12053);
    $j_day_no %= 12053;

    $jy = 979 + 33 * $j_np + 4 * floor($j_day_no / 1461);

    $j_day_no %= 1461;

    if ($j_day_no >= 366) {
        $jy += floor(($j_day_no - 1) / 365);
        $j_day_no = ($j_day_no - 1) % 365;
    }

    $j_month_days = [31, 31, 31, 31, 31, 31, 30, 30, 30, 30, 30, 29];

    for ($i = 0; $i < 11 && $j_day_no >= $j_month_days[$i]; ++$i) {
        $j_day_no -= $j_month_days[$i];
    }

    $jm = $i + 1;
    $jd = $j_day_no + 1;

    return sprintf('%04d/%02d/%02d ⏰ %s', $jy, $jm, $jd, $time);
}

/**
 * Format channel lock statistics for display
 */
function format_channel_statistics(int $channel_chat_id): string {
    $stats = get_channel_statistics($channel_chat_id);

    // Extract channel username from link
    $channel_username = '';
    if (preg_match('/t\.me\/(.+)/', $stats['channel_link'], $matches)) {
        $channel_username = '@' . $matches[1];
    }

    $created_date = $stats['created_at']
        ? '🗓 ' . gregorian_to_jalali($stats['created_at'])
        : 'تاریخ نامشخص';

    $current_date = '🗓 ' . gregorian_to_jalali(date('Y-m-d H:i:s'));

    $text = "╗  🔐 وضعیت این قفل   ╔\n";
    $text .= "║ ‏\n";
    $text .= "╣ ‏ 🔗 {$stats['channel_link']}\n";
    $text .= "╣  ‏🌐 {$channel_username}\n";
    $text .= "║ ‏\n";
    $text .= "╣ از لحظه تنظیم قفل در {$created_date}\n";
    $text .= "║ ‏\n";
    $text .= "╣ ‏👥 {$stats['total_viewed']} نفر ربات را استارت کرده اند؛\n";
    $text .= "║ ‏\n";
    $text .= "╣ ‏ 👨‍👩‍👦 {$stats['joined']} نفر از طریق این قفل، عضو لینک شده اند؛\n";
    $text .= "╣ 🚶‍♂️ {$stats['already_members']} نفر از قبل عضو آن بودند؛\n";
    $text .= "║ ‏\n";
    $text .= "╣  ‏🫣 {$stats['not_joined']} نفر عضو لینک نشده اند / لفت داده اند.\n";
    $text .= "║ ‏\n";
    $text .= $current_date;

    return $text;
}

/**
 * Create channel statistics keyboard
 */
function channel_stats_keyboard(int $channel_chat_id): array {
    return [
        'inline_keyboard' => [
            [['text' => '🔄 به‌روزرسانی', 'callback_data' => "channel_stats_{$channel_chat_id}"]],
            [['text' => '🔙 بازگشت', 'callback_data' => 'admin_lockchannel']]
        ]
    ];
}

/**
 * Create fake channel management keyboard
 */
function fake_channel_keyboard(): array {
    $fake_channels = get_fake_channels();
    $buttons = [];

    foreach ($fake_channels as $fake) {
        $button_text = "📺 {$fake['title']}";
        $buttons[] = [['text' => $button_text, 'callback_data' => "fakechannel_info_{$fake['id']}"]];
    }

    $buttons[] = [['text' => '➕ افزودن کانال جعلی', 'callback_data' => 'fakechannel_add']];
    $buttons[] = [['text' => '🔙 بازگشت', 'callback_data' => 'admin_back']];

    return ['inline_keyboard' => $buttons];
}

/**
 * Create fake channel detail keyboard
 */
function fake_channel_detail_keyboard(int $fake_id): array {
    return [
        'inline_keyboard' => [
            [['text' => 'حذف کانال جعلی', 'callback_data' => "fakechannel_remove_{$fake_id}"]],
            [['text' => '🔙 بازگشت', 'callback_data' => 'admin_fakechannel']]
        ]
    ];
}

/**
 * Create seen/reaction lock management keyboard
 */
function seen_lock_keyboard(): array {
    $locks = get_seen_reaction_locks();
    $lock_count = count($locks);
    $buttons = [];

    foreach ($locks as $lock) {
        $type_emoji = $lock['lock_type'] === 'seen' ? '👁' : '❤️';
        $button_text = "#{$lock['lock_order']} {$type_emoji} قفل {$lock['lock_type']} ({$lock['timer_seconds']}ث)";
        $buttons[] = [['text' => $button_text, 'callback_data' => "seenlock_info_{$lock['id']}"]];
    }

    if ($lock_count < 10) {
        $buttons[] = [['text' => '➕ افزودن قفل', 'callback_data' => 'seenlock_add']];
    } else {
        $buttons[] = [['text' => '⚠️ حداکثر 10 قفل', 'callback_data' => 'noop']];
    }

    $buttons[] = [['text' => '🔙 بازگشت', 'callback_data' => 'admin_back']];

    return ['inline_keyboard' => $buttons];
}

/**
 * Create seen/reaction lock detail keyboard
 */
function seen_lock_detail_keyboard(int $lock_id): array {
    return [
        'inline_keyboard' => [
            [['text' => 'حذف قفل', 'callback_data' => "seenlock_remove_{$lock_id}"]],
            [['text' => '🔙 بازگشت', 'callback_data' => 'admin_seenlock']]
        ]
    ];
}

/**
 * Create lock type selection keyboard
 */
function lock_type_keyboard(): array {
    return [
        'inline_keyboard' => [
            [['text' => '👁 قفل سین', 'callback_data' => 'locktype_seen']],
            [['text' => '❤️ قفل ریاکشن', 'callback_data' => 'locktype_reaction']],
            [['text' => '🔙 لغو', 'callback_data' => 'admin_seenlock']]
        ]
    ];
}

/**
 * Create user lock completion keyboard
 */
function user_lock_keyboard(string $link_code, int $lock_id, bool $can_click): array {
    $button_text = $can_click ? '✅ انجام شد' : '⏳ انجام شد';
    $callback_data = $can_click ? "lock_complete:{$link_code}:{$lock_id}" : "lock_wait:{$link_code}:{$lock_id}";

    return [
        'inline_keyboard' => [
            [['text' => $button_text, 'callback_data' => $callback_data]]
        ]
    ];
}

/**
 * Create link manager keyboard with pagination
 */
function link_manager_keyboard(int $page = 0): array {
    $links = get_all_file_links(10, $page * 10);
    $buttons = [];

    foreach ($links as $link) {
        $button_text = "{$link['link_code']} | {$link['view_count']}";
        $buttons[] = [['text' => $button_text, 'callback_data' => "link_info_{$link['link_code']}"]];
    }

    // Navigation buttons
    $nav_buttons = [];
    if ($page > 0) {
        $nav_buttons[] = ['text' => '◀️ قبلی', 'callback_data' => "link_page_" . ($page - 1)];
    }
    if (count($links) === 10) {
        $nav_buttons[] = ['text' => '▶️ بعدی', 'callback_data' => "link_page_" . ($page + 1)];
    }

    if (!empty($nav_buttons)) {
        $buttons[] = $nav_buttons;
    }

    $buttons[] = [['text' => '🔙 بازگشت', 'callback_data' => 'admin_back']];

    return ['inline_keyboard' => $buttons];
}

/**
 * Create link detail keyboard
 */
function link_detail_keyboard(string $link_code): array {
    return [
        'inline_keyboard' => [
            [['text' => '➕ افزودن فایل', 'callback_data' => "link_addfile_{$link_code}"]],
            [['text' => '🗑 حذف فایل', 'callback_data' => "link_delfile_{$link_code}"]],
            [['text' => '🔙 بازگشت', 'callback_data' => 'admin_link_manager']]
        ]
    ];
}

/**
 * Create file selection keyboard for deletion
 */
function link_file_selection_keyboard(string $link_code, array $files): array {
    $buttons = [];

    for ($i = 0; $i < count($files); $i++) {
        $buttons[] = [['text' => "فایل #" . ($i + 1), 'callback_data' => "link_removefile_{$link_code}_{$i}"]];
    }

    $buttons[] = [['text' => '🔙 بازگشت', 'callback_data' => "link_info_{$link_code}"]];

    return ['inline_keyboard' => $buttons];
}

/**
 * Format link details for display
 */
function format_link_details(string $link_code, array $file_record): string {
    $bot_username = get_bot_username();
    $link = "https://t.me/{$bot_username}?start={$link_code}";
    $file_count = count($file_record['file_data']);

    $text = "🔗 <b>جزئیات لینک</b>\n\n";
    $text .= "📝 کد: <code>{$link_code}</code>\n";
    $text .= "👁 تعداد مشاهده: {$file_record['view_count']}\n";
    $text .= "📦 تعداد فایل: {$file_count}\n";
    $text .= "━━━━━━━━━━━━━━━━━\n";
    $text .= "🔗 لینک:\n{$link}\n";

    return $text;
}

/**
 * Create channel detail keyboard with limits
 */
function channel_detail_keyboard(int $chat_id): array {
    return [
        'inline_keyboard' => [
            [['text' => '📊 مشاهده آمار', 'callback_data' => "channel_stats_{$chat_id}"]],
            [['text' => '⚙️ تنظیم محدودیت', 'callback_data' => "channel_limits_{$chat_id}"]],
            [['text' => 'حذف کانال', 'callback_data' => "channel_remove_{$chat_id}"]],
            [['text' => '🔙 بازگشت', 'callback_data' => 'admin_lockchannel']]
        ]
    ];
}

/**
 * Format channel details with limits
 */
function format_channel_details(array $channel): string {
    $stats = get_channel_statistics($channel['chat_id']);
    $status = is_channel_active($channel['chat_id']) ? '✅ فعال' : '❌ غیرفعال';

    $text = "🔐 <b>جزئیات کانال</b>\n\n";
    $text .= "🆔 Chat ID: <code>{$channel['chat_id']}</code>\n";
    $text .= "🔗 Link: {$channel['channel_link']}\n";
    $text .= "📊 وضعیت: {$status}\n";
    $text .= "━━━━━━━━━━━━━━━━━\n";

    // Join limit
    if ($channel['join_limit'] > 0) {
        $text .= "👥 محدودیت عضو: {$stats['joined']}/{$channel['join_limit']}\n";
    } else {
        $text .= "👥 محدودیت عضو: ∞ (نامحدود)\n";
    }

    // Time limit
    if ($channel['time_limit_hours'] > 0) {
        $created_time = strtotime($channel['created_at']);
        $expire_time = $created_time + ($channel['time_limit_hours'] * 3600);
        $remaining_hours = max(0, floor(($expire_time - time()) / 3600));

        $text .= "⏰ محدودیت زمانی: {$remaining_hours}/{$channel['time_limit_hours']} ساعت\n";
    } else {
        $text .= "⏰ محدودیت زمانی: ∞ (نامحدود)\n";
    }

    return $text;
}

/**
 * Create admin management keyboard
 */
function admin_management_keyboard(): array {
    $dynamic_admins = get_dynamic_admins();
    $buttons = [];

    foreach ($dynamic_admins as $admin) {
        $buttons[] = [['text' => "👤 {$admin['user_id']}", 'callback_data' => "adminmgr_info_{$admin['user_id']}"]];
    }

    $buttons[] = [['text' => '➕ افزودن ادمین', 'callback_data' => 'adminmgr_add']];
    $buttons[] = [['text' => '🔙 بازگشت', 'callback_data' => 'admin_back']];

    return ['inline_keyboard' => $buttons];
}

/**
 * Create admin detail keyboard
 */
function admin_detail_keyboard(int $admin_user_id): array {
    $buttons = [];

    // Only show remove button if not a config admin
    if (!is_config_admin($admin_user_id)) {
        $buttons[] = [['text' => '❌ حذف ادمین', 'callback_data' => "adminmgr_remove_{$admin_user_id}"]];
    } else {
        $buttons[] = [['text' => '🔒 ادمین اصلی (قابل حذف نیست)', 'callback_data' => 'noop']];
    }

    $buttons[] = [['text' => '🔙 بازگشت', 'callback_data' => 'admin_admins']];

    return ['inline_keyboard' => $buttons];
}

/**
 * Extract detailed file information from a message
 * Returns array with file_id, type, caption, and fallback info
 */
function extract_file_info(array $message): array {
    $chat_id = $message['chat']['id'];
    $message_id = $message['message_id'];

    $file_info = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'type' => 'text',
        'file_id' => null,
        'caption' => null,
        'text' => null
    ];

    // Check for photo
    if (isset($message['photo'])) {
        $file_info['type'] = 'photo';
        // Get largest photo
        $photos = $message['photo'];
        $largest = end($photos);
        $file_info['file_id'] = $largest['file_id'];
        $file_info['caption'] = $message['caption'] ?? null;
    }

    // Check for video
    elseif (isset($message['video'])) {
        $file_info['type'] = 'video';
        $file_info['file_id'] = $message['video']['file_id'];
        $file_info['caption'] = $message['caption'] ?? null;
    }

    // Check for document
    elseif (isset($message['document'])) {
        $file_info['type'] = 'document';
        $file_info['file_id'] = $message['document']['file_id'];
        $file_info['caption'] = $message['caption'] ?? null;
        $file_info['file_name'] = $message['document']['file_name'] ?? null;
    }

    // Check for audio
    elseif (isset($message['audio'])) {
        $file_info['type'] = 'audio';
        $file_info['file_id'] = $message['audio']['file_id'];
        $file_info['caption'] = $message['caption'] ?? null;
    }

    // Check for voice
    elseif (isset($message['voice'])) {
        $file_info['type'] = 'voice';
        $file_info['file_id'] = $message['voice']['file_id'];
        $file_info['caption'] = $message['caption'] ?? null;
    }

    // Check for video note
    elseif (isset($message['video_note'])) {
        $file_info['type'] = 'video_note';
        $file_info['file_id'] = $message['video_note']['file_id'];
    }

    // Check for animation (GIF)
    elseif (isset($message['animation'])) {
        $file_info['type'] = 'animation';
        $file_info['file_id'] = $message['animation']['file_id'];
        $file_info['caption'] = $message['caption'] ?? null;
    }

    // Check for sticker
    elseif (isset($message['sticker'])) {
        $file_info['type'] = 'sticker';
        $file_info['file_id'] = $message['sticker']['file_id'];
    }

    // Check for text
    elseif (isset($message['text'])) {
        $file_info['type'] = 'text';
        $file_info['text'] = $message['text'];
    }

    return $file_info;
}

/**
 * Send file using saved file info (fallback when copyMessage fails)
 */
function send_file_by_info(int $chat_id, array $file_info): void {
    // Check if this is new format (has type and file_id) or old format (only chat_id and message_id)
    if (!isset($file_info['type']) || !isset($file_info['file_id'])) {
        // Old format - cannot send without copyMessage, skip silently
        error_log("Cannot send file: old format without file_id (chat_id: {$file_info['chat_id']}, message_id: {$file_info['message_id']})");
        return;
    }

    $type = $file_info['type'];
    $file_id = $file_info['file_id'];
    $caption = $file_info['caption'] ?? null;

    $data = [
        'chat_id' => $chat_id,
        'protect_content' => true
    ];

    if ($caption) {
        $data['caption'] = $caption;
    }

    switch ($type) {
        case 'photo':
            $data['photo'] = $file_id;
            telegram_request('sendPhoto', $data);
            break;

        case 'video':
            $data['video'] = $file_id;
            telegram_request('sendVideo', $data);
            break;

        case 'document':
            $data['document'] = $file_id;
            telegram_request('sendDocument', $data);
            break;

        case 'audio':
            $data['audio'] = $file_id;
            telegram_request('sendAudio', $data);
            break;

        case 'voice':
            $data['voice'] = $file_id;
            telegram_request('sendVoice', $data);
            break;

        case 'video_note':
            $data['video_note'] = $file_id;
            telegram_request('sendVideoNote', $data);
            break;

        case 'animation':
            $data['animation'] = $file_id;
            telegram_request('sendAnimation', $data);
            break;

        case 'sticker':
            $data['sticker'] = $file_id;
            telegram_request('sendSticker', $data);
            break;

        case 'text':
            if (isset($file_info['text'])) {
                $data['text'] = $file_info['text'];
                telegram_request('sendMessage', $data);
            }
            break;
    }
}

/**
 * Get bot setting
 */
function get_bot_setting(string $key, ?string $default = null): ?string {
    $db = get_db();
    $stmt = $db->prepare("SELECT setting_value FROM bot_settings WHERE setting_key = ?");
    $stmt->execute([$key]);
    $result = $stmt->fetchColumn();
    return $result !== false ? $result : $default;
}

/**
 * Set bot setting
 */
function set_bot_setting(string $key, string $value): void {
    $db = get_db();
    $stmt = $db->prepare("INSERT OR REPLACE INTO bot_settings (setting_key, setting_value) VALUES (?, ?)");
    $stmt->execute([$key, $value]);
}

/**
 * Get all subscription plans
 */
function get_subscription_plans(): array {
    $db = get_db();
    $stmt = $db->query("SELECT * FROM subscription_plans ORDER BY price ASC");
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get subscription plan by ID
 */
function get_subscription_plan(int $id): ?array {
    $db = get_db();
    $stmt = $db->prepare("SELECT * FROM subscription_plans WHERE id = ?");
    $stmt->execute([$id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ?: null;
}

/**
 * Add subscription plan
 */
function add_subscription_plan(string $name, float $price, int $duration_days, string $description): void {
    $db = get_db();
    $stmt = $db->prepare("INSERT INTO subscription_plans (name, price, duration_days, description) VALUES (?, ?, ?, ?)");
    $stmt->execute([$name, $price, $duration_days, $description]);
}

/**
 * Delete subscription plan
 */
function delete_subscription_plan(int $id): void {
    $db = get_db();
    $stmt = $db->prepare("DELETE FROM subscription_plans WHERE id = ?");
    $stmt->execute([$id]);
}

/**
 * Get user active subscription
 */
function get_user_subscription(int $user_id): ?array {
    $db = get_db();
    $stmt = $db->prepare("
        SELECT * FROM user_subscriptions 
        WHERE user_id = ? AND status = 'active' AND end_date > CURRENT_TIMESTAMP
        ORDER BY end_date DESC LIMIT 1
    ");
    $stmt->execute([$user_id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ?: null;
}

/**
 * Add user subscription
 */
function add_user_subscription(int $user_id, int $plan_id, int $duration_days): void {
    $db = get_db();
    
    // Calculate end date
    $end_date = date('Y-m-d H:i:s', strtotime("+{$duration_days} days"));
    
    $stmt = $db->prepare("
        INSERT INTO user_subscriptions (user_id, plan_id, end_date, status) 
        VALUES (?, ?, ?, 'active')
    ");
    $stmt->execute([$user_id, $plan_id, $end_date]);
}

/**
 * Create payment request
 */
function create_payment_request(int $user_id, int $plan_id, int $receipt_message_id): int {
    $db = get_db();
    $stmt = $db->prepare("
        INSERT INTO payment_requests (user_id, plan_id, receipt_message_id, status) 
        VALUES (?, ?, ?, 'pending')
    ");
    $stmt->execute([$user_id, $plan_id, $receipt_message_id]);
    return (int)$db->lastInsertId();
}

/**
 * Get payment request
 */
function get_payment_request(int $id): ?array {
    $db = get_db();
    $stmt = $db->prepare("SELECT * FROM payment_requests WHERE id = ?");
    $stmt->execute([$id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ?: null;
}

/**
 * Update payment request status
 */
function update_payment_request_status(int $id, string $status): void {
    $db = get_db();
    $stmt = $db->prepare("UPDATE payment_requests SET status = ? WHERE id = ?");
    $stmt->execute([$status, $id]);
}

/**
 * Admin Subscription Panel Keyboard
 */
function admin_subscription_panel_keyboard(): array {
    $is_enabled = get_bot_setting('subscription_lock_enabled', '0') === '1';
    $toggle_text = $is_enabled ? '✅ قفل فعال است' : '❌ قفل غیرفعال است';
    $toggle_action = $is_enabled ? 'sub_disable' : 'sub_enable';
    
    return [
        'inline_keyboard' => [
            [['text' => $toggle_text, 'callback_data' => $toggle_action]],
            [['text' => '📋 مدیریت پلن‌ها', 'callback_data' => 'sub_plans']],
            [['text' => '💳 تنظیمات کارت بانکی', 'callback_data' => 'sub_bank_info']],
            [['text' => '📢 تنظیم کانال واریزی', 'callback_data' => 'sub_set_channel']],
            [['text' => '🔙 بازگشت', 'callback_data' => 'admin_back']]
        ]
    ];
}

/**
 * Subscription Plans Keyboard
 */
function subscription_plans_keyboard(): array {
    $plans = get_subscription_plans();
    $buttons = [];
    
    foreach ($plans as $plan) {
        $price = number_format($plan['price']);
        $text = "{$plan['name']} - {$price} تومان ({$plan['duration_days']} روز)";
        $buttons[] = [['text' => $text, 'callback_data' => "buy_plan_{$plan['id']}"]];
    }
    
    return ['inline_keyboard' => $buttons];
}

/**
 * Admin Plans Management Keyboard
 */
function admin_plans_keyboard(): array {
    $plans = get_subscription_plans();
    $buttons = [];
    
    foreach ($plans as $plan) {
        $buttons[] = [['text' => "🗑 حذف: {$plan['name']}", 'callback_data' => "sub_del_plan_{$plan['id']}"]];
    }
    
    $buttons[] = [['text' => '➕ افزودن پلن جدید', 'callback_data' => 'sub_add_plan']];
    $buttons[] = [['text' => '🔙 بازگشت', 'callback_data' => 'admin_subscription']];
    
    return ['inline_keyboard' => $buttons];
}

/**
 * Payment Approval Keyboard
 */
function payment_approval_keyboard(int $request_id): array {
    return [
        'inline_keyboard' => [
            [
                ['text' => '✅ تایید', 'callback_data' => "pay_approve_{$request_id}"],
                ['text' => '❌ رد', 'callback_data' => "pay_reject_{$request_id}"]
            ]
        ]
    ];
}
/**
 * Load user states from session file
 */
function load_user_states(): array {
    $file = __DIR__ . '/user_states.json';
    if (file_exists($file)) {
        $content = file_get_contents($file);
        return json_decode($content, true) ?? [];
    }
    return [];
}

/**
 * Save user states to session file
 */
function save_user_states(array $states): void {
    $file = __DIR__ . '/user_states.json';
    file_put_contents($file, json_encode($states, JSON_PRETTY_PRINT));
}

/**
 * Get user state
 */
function get_user_state(int $user_id): ?array {
    $states = load_user_states();
    return $states[$user_id] ?? null;
}

/**
 * Set user state
 */
function set_user_state(int $user_id, array $state): void {
    $states = load_user_states();
    $states[$user_id] = $state;
    save_user_states($states);
}

/**
 * Clear user state
 */
function clear_user_state(int $user_id): void {
    $states = load_user_states();
    unset($states[$user_id]);
    save_user_states($states);
}
?>

