<?php

/**
 * Migration Script for Channel Limits and Dynamic Admins
 * Run this script ONCE to add new columns and tables
 */

require_once 'config.php';
require_once 'database.php';

echo "=== Migration Started: Channel Limits & Admin Management ===\n\n";

try {
    $db = get_db();

    // 1. Add new columns to channels table
    echo "Step 1: Migrating channels table...\n";

    $result = $db->query("PRAGMA table_info(channels)");
    $columns = $result->fetchAll(PDO::FETCH_ASSOC);
    $column_names = array_column($columns, 'name');

    if (!in_array('join_limit', $column_names)) {
        $db->exec("ALTER TABLE channels ADD COLUMN join_limit INTEGER DEFAULT 0");
        echo "✓ Added join_limit column\n";
    } else {
        echo "✓ join_limit column already exists\n";
    }

    if (!in_array('time_limit_hours', $column_names)) {
        $db->exec("ALTER TABLE channels ADD COLUMN time_limit_hours INTEGER DEFAULT 0");
        echo "✓ Added time_limit_hours column\n";
    } else {
        echo "✓ time_limit_hours column already exists\n";
    }

    if (!in_array('is_active', $column_names)) {
        $db->exec("ALTER TABLE channels ADD COLUMN is_active INTEGER DEFAULT 1");
        echo "✓ Added is_active column\n";
    } else {
        echo "✓ is_active column already exists\n";
    }

    // 2. Create dynamic_admins table
    echo "\nStep 2: Creating dynamic_admins table...\n";
    $db->exec("
        CREATE TABLE IF NOT EXISTS dynamic_admins (
            user_id INTEGER PRIMARY KEY,
            added_by INTEGER,
            added_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");
    echo "✓ dynamic_admins table created\n";

    // 3. Add 'left' action tracking support (table already exists, no changes needed)
    echo "\nStep 3: Verifying channel_lock_stats table...\n";
    $result = $db->query("SELECT name FROM sqlite_master WHERE type='table' AND name='channel_lock_stats'");
    if ($result->fetch()) {
        echo "✓ channel_lock_stats table verified (supports 'left' action)\n";
    } else {
        echo "⚠ Warning: channel_lock_stats table not found\n";
    }

    echo "\n=== Migration Completed Successfully ===\n\n";
    echo "New Features Available:\n";
    echo "1. ⚙️ Channel Limits:\n";
    echo "   - Set maximum number of joins per channel\n";
    echo "   - Set time limit (in hours) for channel locks\n";
    echo "   - Channels auto-disable when limits are reached\n\n";
    echo "2. 👥 Admin Management:\n";
    echo "   - Add/remove dynamic admins from the panel\n";
    echo "   - Config admins (in config.php) cannot be removed\n";
    echo "   - Track who added each admin and when\n\n";
    echo "3. 📊 Left User Tracking:\n";
    echo "   - Automatically track users who leave channels\n";
    echo "   - View 'left' count in channel statistics\n";
    echo "   - Requires chat_member updates to be enabled in BotFather\n\n";
    echo "IMPORTANT: Enable chat_member updates (2 steps required):\n\n";
    echo "STEP 1: Update webhook configuration\n";
    echo "  Run: php setup.php\n";
    echo "  This updates the webhook to receive chat_member updates\n\n";
    echo "STEP 2: Enable in BotFather\n";
    echo "  1. Message @BotFather\n";
    echo "  2. Select your bot\n";
    echo "  3. Go to Bot Settings > Group Privacy > Turn OFF\n";
    echo "  4. Go to Bot Settings > Allow Groups > Turn ON\n\n";
    echo "Both steps are REQUIRED for left user tracking to work!\n\n";

} catch (Exception $e) {
    echo "❌ Error during migration: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
    exit(1);
}

?>
