<?php

require_once 'config.php';
require_once 'database.php';
require_once 'functions.php';

echo "=== Telegram Bot Setup ===\n\n";

// Step 1: Initialize database
echo "Step 1: Initializing database...\n";
try {
    init_db();
    echo "✓ Database initialized successfully!\n";
    echo "  Database file: " . DATABASE . "\n\n";
} catch (Exception $e) {
    echo "✗ Error initializing database: " . $e->getMessage() . "\n";
    exit(1);
}

// Step 1.5: Run Migrations
echo "Step 1.5: Running migrations...\n";
$migration_files = glob(__DIR__ . '/migrate*.php');
foreach ($migration_files as $file) {
    $filename = basename($file);
    echo "  Running {$filename}...\n";
    // We use include instead of require to prevent stopping on error if one fails (optional, but safer for re-runs)
    // However, since they might output text, we capture it or just let it print.
    // To avoid variable collisions, we could run them in a separate scope or via shell_exec, 
    // but include is simplest if the scripts are well-behaved.
    // Given the simple nature of these scripts, include is fine.
    try {
        include_once $file;
        echo "  ✓ {$filename} executed.\n";
    } catch (Exception $e) {
        echo "  ✗ Error in {$filename}: " . $e->getMessage() . "\n";
    }
}
echo "\n";

// Step 2: Set webhook
echo "Step 2: Setting webhook...\n";
echo "  Webhook URL: " . WEBHOOK_URL . "\n";

$response = telegram_request('setWebhook', [
    'url' => WEBHOOK_URL,
    'drop_pending_updates' => true,
    'allowed_updates' => json_encode([
        'message',
        'callback_query',
        'chat_member'
    ])
]);

if ($response['ok']) {
    echo "✓ Webhook set successfully!\n";
    echo "  Description: " . ($response['description'] ?? 'N/A') . "\n\n";
} else {
    echo "✗ Error setting webhook: " . ($response['description'] ?? 'Unknown error') . "\n";
    exit(1);
}

// Step 3: Get webhook info
echo "Step 3: Verifying webhook info...\n";
$info = telegram_request('getWebhookInfo');

if ($info['ok']) {
    $webhookInfo = $info['result'];
    echo "✓ Webhook info retrieved:\n";
    echo "  URL: " . ($webhookInfo['url'] ?? 'Not set') . "\n";
    echo "  Pending updates: " . ($webhookInfo['pending_update_count'] ?? 0) . "\n";
    echo "  Allowed updates: " . (isset($webhookInfo['allowed_updates']) ? implode(', ', $webhookInfo['allowed_updates']) : 'All') . "\n";
    echo "  Last error date: " . ($webhookInfo['last_error_date'] ?? 'None') . "\n";
    echo "  Last error message: " . ($webhookInfo['last_error_message'] ?? 'None') . "\n\n";
} else {
    echo "✗ Error getting webhook info\n\n";
}

// Step 4: Get bot info
echo "Step 4: Getting bot info...\n";
$bot_info = telegram_request('getMe');

if ($bot_info['ok']) {
    $bot = $bot_info['result'];
    echo "✓ Bot info:\n";
    echo "  Name: " . ($bot['first_name'] ?? 'N/A') . "\n";
    echo "  Username: @" . ($bot['username'] ?? 'N/A') . "\n";
    echo "  ID: " . ($bot['id'] ?? 'N/A') . "\n\n";
} else {
    echo "✗ Error getting bot info\n\n";
}

// Step 5: Display admin information
echo "Step 5: Configuration summary...\n";
echo "  Admin IDs: " . implode(', ', ADMIN_IDS) . "\n";
echo "  Debug mode: " . (DEBUG_MODE ? 'Enabled' : 'Disabled') . "\n\n";

echo "=== Setup Complete! ===\n";
echo "\nYour bot is now ready to receive updates at:\n";
echo WEBHOOK_URL . "\n\n";

echo "Next steps:\n";
echo "1. Make sure your webhook.php file is accessible at the URL above\n";
echo "2. Send /start to your bot to test it\n";
echo "3. Check the error.log and updates.log files for debugging\n";
echo "4. For security, delete or restrict access to this setup.php file after setup\n\n";

?>

