<?php

/**
 * Test Cron Job Script
 * Run this manually to test if your cron environment is set up correctly
 * Usage: php test_cron.php
 */

echo "=== Cron Job Test ===\n\n";

// Test 1: PHP Version
echo "1. PHP Version: " . PHP_VERSION . "\n";
if (version_compare(PHP_VERSION, '8.1.0', '>=')) {
    echo "   ✓ OK (8.1+)\n\n";
} else {
    echo "   ✗ FAIL (Requires 8.1+)\n\n";
}

// Test 2: File Paths
echo "2. File Paths:\n";
echo "   Script directory: " . __DIR__ . "\n";
echo "   Config file: " . (__DIR__ . '/config.php') . "\n";
echo "   Config exists: " . (file_exists(__DIR__ . '/config.php') ? '✓ Yes' : '✗ No') . "\n\n";

// Test 3: Load Config
echo "3. Loading Config:\n";
if (file_exists(__DIR__ . '/config.php')) {
    require_once __DIR__ . '/config.php';
    echo "   ✓ Config loaded\n";
    echo "   Database: " . DATABASE . "\n";
    echo "   Bot Token: " . substr(BOT_TOKEN, 0, 10) . "...\n\n";
} else {
    echo "   ✗ Config not found\n\n";
    exit(1);
}

// Test 4: Database Access
echo "4. Database Access:\n";
require_once __DIR__ . '/database.php';
try {
    $db = get_db();
    echo "   ✓ Database connection OK\n";
    
    // Check if broadcast_queue table exists
    $result = $db->query("SELECT name FROM sqlite_master WHERE type='table' AND name='broadcast_queue'");
    if ($result->fetch()) {
        echo "   ✓ broadcast_queue table exists\n";
        
        // Count pending broadcasts
        $stmt = $db->query("SELECT COUNT(*) FROM broadcast_queue WHERE status='pending'");
        $count = $stmt->fetchColumn();
        echo "   Pending broadcasts: $count\n\n";
    } else {
        echo "   ✗ broadcast_queue table not found\n";
        echo "   Run setup.php first!\n\n";
    }
} catch (Exception $e) {
    echo "   ✗ Database error: " . $e->getMessage() . "\n\n";
}

// Test 5: Functions
echo "5. Functions:\n";
require_once __DIR__ . '/functions.php';
echo "   ✓ Functions loaded\n";

// Test API connection
$response = telegram_request('getMe');
if (isset($response['ok']) && $response['ok']) {
    echo "   ✓ Telegram API connection OK\n";
    echo "   Bot: @" . $response['result']['username'] . "\n\n";
} else {
    echo "   ✗ Telegram API error\n";
    echo "   Response: " . json_encode($response) . "\n\n";
}

// Test 6: Write Permission
echo "6. Write Permission:\n";
$test_file = __DIR__ . '/test_write.tmp';
if (file_put_contents($test_file, 'test')) {
    echo "   ✓ Can write to directory\n";
    unlink($test_file);
} else {
    echo "   ✗ Cannot write to directory\n";
}
echo "\n";

// Test 7: Cron Job Simulation
echo "7. Simulating Cron Job:\n";
echo "   This is what the cron job would do...\n";

$pending = get_pending_broadcasts();
if (empty($pending)) {
    echo "   ℹ No pending broadcasts (this is normal)\n";
} else {
    echo "   Found " . count($pending) . " pending broadcast(s)\n";
    foreach ($pending as $b) {
        echo "   - Broadcast #{$b['id']}: {$b['broadcast_type']}, {$b['total_users']} users\n";
    }
}
echo "\n";

// Summary
echo "=== Summary ===\n";
echo "If all tests passed, your cron job should work!\n\n";
echo "To set up the cron job:\n";
echo "1. Run: crontab -e\n";
echo "2. Add: */3 * * * * /usr/bin/php " . __DIR__ . "/broadcast_cron.php\n";
echo "3. Save and exit\n\n";
echo "To test the actual cron script:\n";
echo "php " . __DIR__ . "/broadcast_cron.php\n\n";

?>

