<?php

require_once 'config.php';
require_once 'database.php';
require_once 'functions.php';

// Get incoming update from Telegram
$content = file_get_contents('php://input');
$update = json_decode($content, true);

// Log the update for debugging
if (DEBUG_MODE) {
    file_put_contents(__DIR__ . '/updates.log', date('Y-m-d H:i:s') . " - " . $content . "\n", FILE_APPEND);
}

// Exit if no update received
if (!$update) {
    http_response_code(200);
    exit;
}

// Handle different update types
if (isset($update['message'])) {
    handle_message($update['message']);
} elseif (isset($update['callback_query'])) {
    handle_callback_query($update['callback_query']);
} elseif (isset($update['chat_member'])) {
    handle_chat_member_update($update['chat_member']);
}

http_response_code(200);

/**
 * Handle incoming messages
 */
function handle_message(array $message): void {
    $user_id = $message['from']['id'];
    $chat_id = $message['chat']['id'];
    $text = $message['text'] ?? '';
    
    // Handle /start command
    if (isset($message['entities']) && $message['entities'][0]['type'] === 'bot_command' && strpos($text, '/start') === 0) {
        handle_start_command($message);
        return;
    }
    
    // Handle /cancel command for admins
    if ($text === '/cancel' && is_admin($user_id)) {
        $state = get_admin_state($user_id);
        if ($state) {
            $mode = $state['mode'] ?? '';
            clear_admin_state($user_id);

            if ($mode === 'add_channel') {
                send_message($chat_id, 'عملیات لغو شد.', channel_lock_keyboard());
            } elseif ($mode === 'add_fake_channel') {
                send_message($chat_id, 'عملیات لغو شد.', fake_channel_keyboard());
            } elseif ($mode === 'add_seenlock') {
                send_message($chat_id, 'عملیات لغو شد.', seen_lock_keyboard());
            } elseif ($mode === 'link_addfile') {
                send_message($chat_id, 'عملیات لغو شد.', link_manager_keyboard(0));
            } elseif ($mode === 'set_channel_limits') {
                $channel = get_channel_by_id($state['chat_id']);
                send_message($chat_id, 'عملیات لغو شد.', channel_detail_keyboard($state['chat_id']));
            } elseif ($mode === 'add_admin') {
                send_message($chat_id, 'عملیات لغو شد.', admin_management_keyboard());
            } else {
                send_message($chat_id, 'عملیات لغو شد.', admin_panel_keyboard());
            }
        }
        return;
    }
    
    // Handle /done command for admins
    if ($text === '/done' && is_admin($user_id)) {
        $state = get_admin_state($user_id);
        if ($state && $state['mode'] === 'upload') {
            if (empty($state['files'])) {
                send_message($chat_id, 'هیچ فایلی دریافت نشده است.');
                return;
            }

            $link_code = generate_link_code();
            add_file_record($link_code, $state['files']);

            $bot_username = get_bot_username();
            $link = "https://t.me/{$bot_username}?start={$link_code}";

            send_message($chat_id, "فایل(ها) ذخیره شدند.\nلینک دریافت فایل:\n{$link}");
            clear_admin_state($user_id);
            send_message($chat_id, 'پنل ادمین:', admin_panel_keyboard());
        } elseif ($state && $state['mode'] === 'link_addfile') {
            if (empty($state['files'])) {
                send_message($chat_id, 'هیچ فایلی دریافت نشده است.');
                return;
            }

            $link_code = $state['link_code'];
            $file_record = get_file_record($link_code);

            if ($file_record) {
                // Merge new files with existing files
                $updated_files = array_merge($file_record['file_data'], $state['files']);
                update_file_data($link_code, $updated_files);

                send_message($chat_id, "✅ فایل‌های جدید به لینک اضافه شدند.\nتعداد فایل جدید: " . count($state['files']));
                clear_admin_state($user_id);

                // Show updated link details
                $updated_record = get_file_record($link_code);
                $text = format_link_details($link_code, $updated_record);
                send_message($chat_id, $text, link_detail_keyboard($link_code));
            } else {
                send_message($chat_id, 'خطا: لینک یافت نشد.');
                clear_admin_state($user_id);
            }
        }
        return;
    }
    
    // Handle admin sending bot links to edit (without being in a state)
    if (is_admin($user_id) && preg_match('/[?&]start=([a-zA-Z0-9]+)/', $text, $matches)) {
        $link_code = $matches[1];
        $file_record = get_file_record($link_code);

        if ($file_record) {
            // Show link details directly
            $link_details = format_link_details($link_code, $file_record);
            send_message($chat_id, $link_details, link_detail_keyboard($link_code));
            return;
        } else {
            send_message($chat_id, 'لینک یافت نشد.');
            return;
        }
    }

    // Handle admin state-based messages
    if (is_admin($user_id)) {
        handle_admin_state_message($message);
    }
}

/**
 * Handle /start command
 */
function handle_start_command(array $message): void {
    $user_id = $message['from']['id'];
    $chat_id = $message['chat']['id'];
    $text = $message['text'];
    
    // Add user to database if not admin
    if (!is_admin($user_id)) {
        add_user($user_id);
    }
    
    // Extract payload (link code) from /start command
    $parts = explode(' ', $text, 2);
    
    if (count($parts) > 1) {
        $payload = $parts[1];
        $file_record = get_file_record($payload);
        
        if (!$file_record) {
            send_message($chat_id, 'لینک نامعتبر است!');
            return;
        }

        // Check for subscription lock
        $is_lock_enabled = get_bot_setting('subscription_lock_enabled', '0') === '1';
        if ($is_lock_enabled) {
            $subscription = get_user_subscription($user_id);
            if (!$subscription) {
                // User has no active subscription
                send_message(
                    $chat_id,
                    "⛔️ برای دسترسی به این فایل باید اشتراک تهیه کنید.\n\nلطفاً یکی از پلن‌های زیر را انتخاب کنید:",
                    subscription_plans_keyboard()
                );
                return;
            }
        }

        // Start user session for timer
        start_user_session($user_id, $payload);

        // Check if user is member of all required channels
        $not_member = check_user_membership($user_id);

        if (!empty($not_member)) {
            // Track that user viewed channel lock requirement
            foreach ($not_member as $channel) {
                track_channel_interaction($user_id, $channel['chat_id'], 'viewed');
            }

            send_message(
                $chat_id,
                'برای دریافت فایل باید در کانال‌های زیر عضو شوید:',
                user_join_keyboard($payload)
            );
            return;
        }

        // Track already members (user was already member of all channels)
        $all_channels = get_channels();
        foreach ($all_channels as $channel) {
            track_channel_interaction($user_id, $channel['chat_id'], 'already_member');
        }

        // Check for seen/reaction locks
        $next_lock = get_next_user_lock($user_id, $payload);

        if ($next_lock) {
            // Start first lock
            start_user_lock($user_id, $payload, $next_lock['id']);

            $type_fa = $next_lock['lock_type'] === 'seen' ? 'سین' : 'ریاکشن';
            send_message(
                $chat_id,
                "قفل {$type_fa} #{$next_lock['lock_order']}:\n\n{$next_lock['message_text']}",
                user_lock_keyboard($payload, $next_lock['id'], false)
            );
            return;
        }

        // Send files to user (no locks)
        send_files_to_user($chat_id, $user_id, $payload, $file_record);
        
    } else {
        // No payload - show admin panel or welcome message
        if (is_admin($user_id)) {
            send_message($chat_id, 'پنل ادمین:', admin_panel_keyboard());
        } else {
            send_message($chat_id, 'سلام! برای دریافت فایل، لطفاً از لینک معتبر استفاده کنید.');
        }
    }
}

/**
 * Send files to user
 */
function send_files_to_user(int $chat_id, int $user_id, string $link_code, array $file_record): void {
    $file_data = $file_record['file_data'];

    foreach ($file_data as $item) {
        // First try copyMessage
        $result = copy_message($chat_id, $item['chat_id'], $item['message_id']);

        // If copyMessage failed, fall back to sending by file_id
        if (!$result) {
            send_file_by_info($chat_id, $item);
        }
    }

    increment_view_count($link_code);

    send_message(
        $chat_id,
        "فایل ارسال شد. تعداد مشاهده: " . ($file_record['view_count'] + 1),
        file_sent_keyboard($link_code, $user_id)
    );
}

/**
 * Handle admin state-based messages
 */
function handle_admin_state_message(array $message): void {
    $user_id = $message['from']['id'];
    $chat_id = $message['chat']['id'];
    $message_id = $message['message_id'];
    $text = $message['text'] ?? '';
    
    $state = get_admin_state($user_id);
    if (!$state) {
        return;
    }
    
    $mode = $state['mode'];
    
    // Handle upload mode
    if ($mode === 'upload') {
        // Extract detailed file information
        $file_info = extract_file_info($message);
        $state['files'][] = $file_info;
        set_admin_state($user_id, $state);
        send_message($chat_id, 'فایل ذخیره شد.');
    }

    // Handle link_addfile mode
    elseif ($mode === 'link_addfile') {
        // Check if message is a bot link for editing
        if (preg_match('/[?&]start=([a-zA-Z0-9]+)/', $text, $matches)) {
            $link_code = $matches[1];
            $file_record = get_file_record($link_code);

            if ($file_record) {
                // Switch to editing this link
                $state['link_code'] = $link_code;
                $state['files'] = [];
                set_admin_state($user_id, $state);

                $link_details = format_link_details($link_code, $file_record);
                send_message($chat_id, "✅ لینک پیدا شد!\n\n" . $link_details . "\n\nحالا فایل‌های جدید را ارسال کنید یا /done را بزنید.");
            } else {
                send_message($chat_id, 'لینک یافت نشد. لطفاً لینک معتبر ارسال کنید یا فایل‌ها را ارسال کنید.');
            }
        } else {
            // It's a file, add to the list
            if (!isset($state['files'])) {
                $state['files'] = [];
            }

            // Extract detailed file information
            $file_info = extract_file_info($message);
            $state['files'][] = $file_info;
            set_admin_state($user_id, $state);
            send_message($chat_id, 'فایل ذخیره شد.');
        }
    }
    
    // Handle broadcast modes
    elseif ($mode === 'broadcast_copy' || $mode === 'broadcast_forward') {
        // Queue the broadcast instead of sending immediately
        $message_data = [
            'chat_id' => $chat_id,
            'message_id' => $message_id
        ];
        
        $broadcast_type = ($mode === 'broadcast_copy') ? 'copy' : 'forward';
        $broadcast_id = queue_broadcast($message_data, $broadcast_type);
        
        $total_users = get_user_count();
        
        send_message(
            $chat_id, 
            "✅ پیام شما در صف ارسال قرار گرفت.\n\n" .
            "🆔 شناسه پخش: #{$broadcast_id}\n" .
            "👥 تعداد کاربران: {$total_users}\n" .
            "⏱ ارسال توسط Cron Job انجام خواهد شد.\n\n" .
            "📊 برای مشاهده وضعیت، از پنل ادمین استفاده کنید."
        );
        
        clear_admin_state($user_id);
        send_message($chat_id, 'پنل ادمین:', admin_panel_keyboard());
    }
    
    // Handle add channel mode
    elseif ($mode === 'add_channel') {
        $step = $state['step'];

        if ($step === 'chat_id') {
            if (is_numeric($text)) {
                $channel_chat_id = (int)$text;

                // Verify bot has access to the channel
                $chat = get_chat($channel_chat_id);

                if ($chat) {
                    $state['chat_id'] = $channel_chat_id;
                    $state['step'] = 'channel_link';
                    set_admin_state($user_id, $state);

                    send_message(
                        $chat_id,
                        "Chat ID تأیید شد. حالا لطفاً لینک کانال را ارسال کنید.\nمثال: https://t.me/yourchannel یا @yourchannel"
                    );
                } else {
                    send_message(
                        $chat_id,
                        "خطا: ربات به این کانال دسترسی ندارد.\n" .
                        "لطفاً مطمئن شوید که:\n" .
                        "1. Chat ID صحیح است\n" .
                        "2. ربات ادمین کانال است\n" .
                        "3. ربات به کانال دسترسی دارد\n\n" .
                        "لطفاً دوباره Chat ID را ارسال کنید یا /cancel را بزنید."
                    );
                }
            } else {
                send_message($chat_id, "لطفاً یک Chat ID معتبر ارسال کنید.\nمثال: -1001234567890");
            }
        }

        elseif ($step === 'channel_link') {
            if (preg_match('/^(https?:\/\/t\.me\/|@)/', $text)) {
                $channel_link = $text;

                if (strpos($channel_link, '@') === 0) {
                    $channel_link = 'https://t.me/' . substr($channel_link, 1);
                }

                add_channel($state['chat_id'], $channel_link);
                send_message($chat_id, '✅ کانال با موفقیت اضافه شد!');
                clear_admin_state($user_id);
                send_message($chat_id, 'مدیریت قفل کانال:', channel_lock_keyboard());
            } else {
                send_message(
                    $chat_id,
                    "لطفاً یک لینک معتبر کانال ارسال کنید.\nمثال: https://t.me/yourchannel یا @yourchannel"
                );
            }
        }
    }

    // Handle add fake channel mode
    elseif ($mode === 'add_fake_channel') {
        $step = $state['step'];

        if ($step === 'title') {
            // Save title and ask for link
            $state['title'] = $text;
            $state['step'] = 'link';
            set_admin_state($user_id, $state);

            send_message(
                $chat_id,
                "عنوان ذخیره شد. حالا لطفاً لینک کانال جعلی را ارسال کنید.\nمثال: https://t.me/fakechannel یا @fakechannel"
            );
        }

        elseif ($step === 'link') {
            if (preg_match('/^(https?:\/\/t\.me\/|@)/', $text)) {
                $channel_link = $text;

                if (strpos($channel_link, '@') === 0) {
                    $channel_link = 'https://t.me/' . substr($channel_link, 1);
                }

                add_fake_channel($state['title'], $channel_link);
                send_message($chat_id, '✅ کانال جعلی با موفقیت اضافه شد!');
                clear_admin_state($user_id);
                send_message($chat_id, '🎭 مدیریت کانال‌های جعلی:', fake_channel_keyboard());
            } else {
                send_message(
                    $chat_id,
                    "لطفاً یک لینک معتبر ارسال کنید.\nمثال: https://t.me/fakechannel یا @fakechannel"
                );
            }
        }
    }

    // Handle add seen/reaction lock mode
    elseif ($mode === 'add_seenlock') {
        $step = $state['step'];

        if ($step === 'message') {
            // Save message and ask for timer
            $state['message'] = $text;
            $state['step'] = 'timer';
            set_admin_state($user_id, $state);

            $type_fa = $state['lock_type'] === 'seen' ? 'سین' : 'ریاکشن';
            send_message(
                $chat_id,
                "متن ذخیره شد.\n\nحالا مدت زمان تایمر را به ثانیه وارد کنید.\n(مثال: 30)"
            );
        }

        elseif ($step === 'timer') {
            if (is_numeric($text) && (int)$text > 0) {
                $timer_seconds = (int)$text;

                // Add the lock
                add_seen_reaction_lock($state['lock_type'], $state['message'], $timer_seconds);

                $type_fa = $state['lock_type'] === 'seen' ? 'سین' : 'ریاکشن';
                send_message($chat_id, "✅ قفل {$type_fa} با موفقیت اضافه شد!");

                clear_admin_state($user_id);
                send_message($chat_id, '👁 مدیریت قفل سین/ریاکشن:', seen_lock_keyboard());
            } else {
                send_message($chat_id, "لطفاً یک عدد معتبر وارد کنید.\nمثال: 30");
            }
        }
    }

    // Handle set channel limits mode
    elseif ($mode === 'set_channel_limits') {
        $step = $state['step'];

        if ($step === 'join_limit') {
            if (is_numeric($text) && (int)$text >= 0) {
                $state['join_limit'] = (int)$text;
                $state['step'] = 'time_limit';
                set_admin_state($user_id, $state);

                send_message(
                    $chat_id,
                    "✅ محدودیت عضو ذخیره شد.\n\n⏰ حالا محدودیت زمانی را به ساعت وارد کنید:\n(برای نامحدود عدد 0 وارد کنید)"
                );
            } else {
                send_message($chat_id, "لطفاً یک عدد معتبر وارد کنید.\nمثال: 100");
            }
        }

        elseif ($step === 'time_limit') {
            if (is_numeric($text) && (int)$text >= 0) {
                $time_limit_hours = (int)$text;

                // Update channel limits
                update_channel_limits($state['chat_id'], $state['join_limit'], $time_limit_hours);

                send_message($chat_id, "✅ محدودیت‌ها با موفقیت تنظیم شد!");

                clear_admin_state($user_id);

                // Show updated channel details
                $channel = get_channel_by_id($state['chat_id']);
                $text = format_channel_details($channel);
                send_message($chat_id, $text, channel_detail_keyboard($state['chat_id']));
            } else {
                send_message($chat_id, "لطفاً یک عدد معتبر وارد کنید.\nمثال: 24");
            }
        }
    }

    // Handle add admin mode
    elseif ($mode === 'add_admin') {
        if (is_numeric($text)) {
            $new_admin_id = (int)$text;

            if (add_dynamic_admin($new_admin_id, $user_id)) {
                send_message($chat_id, "✅ ادمین جدید با موفقیت اضافه شد!\nUser ID: {$new_admin_id}");
                clear_admin_state($user_id);
                send_message($chat_id, '👥 مدیریت ادمین‌ها:', admin_management_keyboard());
            } else {
                send_message($chat_id, "❌ خطا: این کاربر قبلاً ادمین است یا مشکلی پیش آمده.");
            }
        } else {
            send_message($chat_id, "لطفاً یک User ID معتبر (عدد) ارسال کنید.");
        }
    }

    // Handle subscription admin modes
    elseif (strpos($mode, 'sub_') === 0) {
        handle_admin_subscription_message($message, $state);
    }

    // Handle user waiting for receipt
    elseif ($mode === 'waiting_for_receipt') {
        if (isset($message['photo'])) {
            $plan_id = $state['plan_id'];
            $plan = get_subscription_plan($plan_id);
            
            if ($plan) {
                // Forward receipt to config channel
                $config_channel_id = (int)get_bot_setting('payment_channel_id', PAYMENT_CONFIG_CHAT_ID);
                
                if ($config_channel_id == 0) {
                    send_message($chat_id, "❌ خطا: کانال واریزی تنظیم نشده است. لطفاً به ادمین اطلاع دهید.");
                    clear_user_state($user_id);
                    return;
                }

                // Send receipt to admin channel with approval buttons
                $caption = "🧾 درخواست اشتراک جدید\n\n";
                $caption .= "👤 کاربر: {$user_id}\n";
                $caption .= "📦 پلن: {$plan['name']}\n";
                $caption .= "💰 مبلغ: " . number_format($plan['price']) . " تومان\n";
                $caption .= "📅 مدت: {$plan['duration_days']} روز";

                $photo_id = end($message['photo'])['file_id'];
                
                $response = telegram_request('sendPhoto', [
                    'chat_id' => $config_channel_id,
                    'photo' => $photo_id,
                    'caption' => $caption
                ]);

                if (isset($response['result']['message_id'])) {
                    $receipt_message_id = $response['result']['message_id'];
                    
                    // Create payment request
                    $request_id = create_payment_request($user_id, $plan_id, $receipt_message_id);
                    
                    // Add approval buttons to the admin message
                    edit_message_reply_markup(
                        $config_channel_id, 
                        $receipt_message_id, 
                        payment_approval_keyboard($request_id)
                    );

                    send_message($chat_id, "✅ رسید شما دریافت شد و پس از تایید ادمین، اشتراک شما فعال می‌شود.");
                    clear_user_state($user_id);
                } else {
                    send_message($chat_id, "❌ خطا در ارسال رسید. لطفاً دوباره تلاش کنید.");
                }
            } else {
                send_message($chat_id, "❌ خطا: پلن یافت نشد.");
                clear_user_state($user_id);
            }
        } else {
            send_message($chat_id, "لطفاً تصویر رسید را ارسال کنید (یا /cancel را بزنید).");
        }
    }
}

/**
 * Handle callback queries (inline button clicks)
 */
function handle_callback_query(array $callback_query): void {
    $callback_id = $callback_query['id'];
    $user_id = $callback_query['from']['id'];
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $data = $callback_query['data'];
    
    // Handle like button
    if (strpos($data, 'like:') === 0) {
        $link_code = substr($data, 5);
        $like_status = has_user_liked($link_code, $user_id);
        
        if ($like_status) {
            if (remove_like($link_code, $user_id)) {
                answer_callback_query($callback_id, 'لایک شما حذف شد');
            }
        } else {
            if (add_like($link_code, $user_id)) {
                answer_callback_query($callback_id, 'با تشکر از لایک شما!');
            } else {
                answer_callback_query($callback_id, 'قبلا لایک کرده‌اید!');
            }
        }
        
        edit_message_reply_markup($chat_id, $message_id, file_sent_keyboard($link_code, $user_id));
        return;
    }
   // devLog($data);
    // Handle admin management callbacks
    if (strpos($data, 'adminmgr_') === 0 || $data === 'admin_admins') {
     //   devLog("data");
        handle_adminmgr_callback($callback_query);
        return;
    }
    // Handle admin panel callbacks
    if (strpos($data, 'admin_') === 0) {
        if (!is_admin($user_id)) {
            answer_callback_query($callback_id, 'شما اجازه دسترسی به این بخش را ندارید.', true);
            return;
        }
        
        handle_admin_callback($callback_query);
        return;
    }
    
    // Handle join check callback
    if (strpos($data, 'join_check:') === 0) {
        $link_code = substr($data, 11);

        // Check session elapsed time (20 second requirement)
        $elapsed = get_session_elapsed_time($user_id, $link_code);

        // Check real channel membership
        $not_member = check_user_membership($user_id);

        // Require both: 20 seconds elapsed AND all real channels joined
        if ($elapsed < 20 || !empty($not_member)) {
            answer_callback_query($callback_id, 'هنوز عضو همه کانال‌ها نشده‌اید.', true);
        } else {
            $file_record = get_file_record($link_code);

            if (!$file_record) {
                answer_callback_query($callback_id, 'لینک نامعتبر است.', true);
                return;
            }

            // Track that user joined (they were not member before, but now they are)
            $all_channels = get_channels();
            foreach ($all_channels as $channel) {
                track_channel_interaction($user_id, $channel['chat_id'], 'joined');
            }

            // Clear session after successful access
            clear_user_session($user_id, $link_code);

            // Check for seen/reaction locks
            $next_lock = get_next_user_lock($user_id, $link_code);

            if ($next_lock) {
                // Start first lock
                start_user_lock($user_id, $link_code, $next_lock['id']);

                $type_fa = $next_lock['lock_type'] === 'seen' ? 'سین' : 'ریاکشن';
                send_message(
                    $chat_id,
                    "قفل {$type_fa} #{$next_lock['lock_order']}:\n\n{$next_lock['message_text']}",
                    user_lock_keyboard($link_code, $next_lock['id'], false)
                );
                answer_callback_query($callback_id, 'به قفل بعدی منتقل شدید.');
                return;
            }

            send_files_to_user($chat_id, $user_id, $link_code, $file_record);
            answer_callback_query($callback_id, 'فایل ارسال شد.');
        }
        return;
    }

    
    // Handle channel management callbacks
    if (strpos($data, 'channel_') === 0) {
        handle_channel_callback($callback_query);
        return;
    }

    // Handle fake channel management callbacks
    if (strpos($data, 'fakechannel_') === 0) {
        handle_fakechannel_callback($callback_query);
        return;
    }

    // Handle seen/reaction lock callbacks
    if (strpos($data, 'seenlock_') === 0 || strpos($data, 'locktype_') === 0) {
        handle_seenlock_callback($callback_query);
        return;
    }

    // Handle user lock completion callbacks
    if (strpos($data, 'lock_') === 0) {
        handle_user_lock_callback($callback_query);
        return;
    }

    // Handle broadcast management callbacks
    if (strpos($data, 'broadcast_') === 0) {
        handle_broadcast_callback($callback_query);
        return;
    }

    // Handle link management callbacks
    if (strpos($data, 'link_') === 0) {
        handle_link_callback($callback_query);
        return;
    }

    // Handle subscription admin callbacks
    if (strpos($data, 'sub_') === 0 || $data === 'admin_subscription') {
        handle_admin_subscription_callback($callback_query);
        return;
    }

    // Handle user buying plan
    if (strpos($data, 'buy_plan_') === 0) {
        $plan_id = (int)substr($data, 9);
        $plan = get_subscription_plan($plan_id);
        
        if ($plan) {
            $bank_info = get_bot_setting('bank_info', 'تنظیم نشده');
            $price = number_format($plan['price']);
            
            $text = "💳 خرید اشتراک: {$plan['name']}\n\n";
            $text .= "💰 مبلغ قابل پرداخت: {$price} تومان\n\n";
            $text .= "💳 شماره کارت:\n<code>{$bank_info}</code>\n\n";
            $text .= "لطفاً مبلغ را به کارت بالا واریز کرده و تصویر رسید را ارسال کنید.";
            
            set_user_state($user_id, ['mode' => 'waiting_for_receipt', 'plan_id' => $plan_id]);
            
            send_message($chat_id, $text);
            answer_callback_query($callback_id);
        } else {
            answer_callback_query($callback_id, 'پلن یافت نشد.', true);
        }
        return;
    }

    // Handle payment approval/rejection (in config channel)
    if (strpos($data, 'pay_') === 0) {
        handle_payment_callback($callback_query);
        return;
    }

}

/**
 * Handle admin panel callbacks
 */
function handle_admin_callback(array $callback_query): void {
    $callback_id = $callback_query['id'];
    $user_id = $callback_query['from']['id'];
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $data = $callback_query['data'];
    
    if ($data === 'admin_upload') {
        set_admin_state($user_id, ['mode' => 'upload', 'files' => []]);
        edit_message_text(
            $chat_id,
            $message_id,
            'لطفا فایل (یا پیام) مورد نظر برای آپلود را ارسال کنید. پس از اتمام، دستور /done را بفرستید.'
        );
    }
    
    elseif ($data === 'admin_stats') {
        $count = get_user_count();
        answer_callback_query($callback_id, "تعداد کاربران: {$count}", true);
    }
    
    elseif ($data === 'admin_broadcast_copy') {
        set_admin_state($user_id, ['mode' => 'broadcast_copy']);
        edit_message_text($chat_id, $message_id, 'پیام مورد نظر را برای ارسال همگانی (copy) ارسال کنید.');
    }
    
    elseif ($data === 'admin_broadcast_forward') {
        set_admin_state($user_id, ['mode' => 'broadcast_forward']);
        edit_message_text($chat_id, $message_id, 'پیام مورد نظر را برای ارسال همگانی (forward) ارسال کنید.');
    }
    
    elseif ($data === 'admin_lockchannel') {
        edit_message_text($chat_id, $message_id, 'مدیریت قفل کانال:', channel_lock_keyboard());
    }

    elseif ($data === 'admin_broadcast_manager') {
        edit_message_text($chat_id, $message_id, '📡 مدیریت پخش‌های همگانی:', broadcast_manager_keyboard(0));
    }

    elseif ($data === 'admin_back') {
        edit_message_text($chat_id, $message_id, 'پنل ادمین:', admin_panel_keyboard());
    }

    elseif ($data === 'admin_fakechannel') {
        edit_message_text($chat_id, $message_id, '🎭 مدیریت کانال‌های جعلی:', fake_channel_keyboard());
    }

    elseif ($data === 'admin_seenlock') {
        edit_message_text($chat_id, $message_id, '👁 مدیریت قفل سین/ریاکشن:', seen_lock_keyboard());
    }

    elseif ($data === 'admin_link_manager') {
        edit_message_text($chat_id, $message_id, '🔗 مدیریت لینک‌های صادر شده:', link_manager_keyboard(0));
    }

    answer_callback_query($callback_id);
}

/**
 * Handle channel management callbacks
 */
function handle_channel_callback(array $callback_query): void {
    $callback_id = $callback_query['id'];
    $user_id = $callback_query['from']['id'];
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $data = $callback_query['data'];

    // Handle channel statistics view
    if (strpos($data, 'channel_stats_') === 0) {
        $channel_chat_id = (int)substr($data, 14);

        $text = format_channel_statistics($channel_chat_id);
        $keyboard = channel_stats_keyboard($channel_chat_id);
        edit_message_text($chat_id, $message_id, $text, $keyboard);
        answer_callback_query($callback_id, 'آمار به‌روزرسانی شد.');
        return;
    }

    if (strpos($data, 'channel_info_') === 0) {
        $channel_chat_id = (int)substr($data, 13);
        $channel = get_channel_by_id($channel_chat_id);

        if ($channel) {
            $text = format_channel_details($channel);
            $keyboard = channel_detail_keyboard($channel_chat_id);
            edit_message_text($chat_id, $message_id, $text, $keyboard);
            answer_callback_query($callback_id);
        } else {
            answer_callback_query($callback_id, 'کانال یافت نشد.', true);
        }
    }

    // Handle channel limits callback
    elseif (strpos($data, 'channel_limits_') === 0) {
        handle_channel_callback_extended($callback_query);
        return;
    }
    
    elseif (strpos($data, 'channel_remove_') === 0) {
        $channel_chat_id = (int)substr($data, 15);
        remove_channel($channel_chat_id);
        answer_callback_query($callback_id, 'کانال حذف شد.', true);
        edit_message_text($chat_id, $message_id, 'کانال حذف شد.', channel_lock_keyboard());
    }
    
    elseif ($data === 'channel_add') {
        set_admin_state($user_id, ['mode' => 'add_channel', 'step' => 'chat_id']);
        edit_message_text(
            $chat_id,
            $message_id,
            "لطفاً Chat ID کانال را ارسال کنید.\nمثال: -1001234567890"
        );
    }
}

/**
 * Handle fake channel management callbacks
 */
function handle_fakechannel_callback(array $callback_query): void {
    $callback_id = $callback_query['id'];
    $user_id = $callback_query['from']['id'];
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $data = $callback_query['data'];

    // Check admin permission
    if (!is_admin($user_id)) {
        answer_callback_query($callback_id, 'شما اجازه دسترسی به این بخش را ندارید.', true);
        return;
    }

    // Handle fake channel info display
    if (strpos($data, 'fakechannel_info_') === 0) {
        $fake_id = (int)substr($data, 17);
        $fake = get_fake_channel_by_id($fake_id);

        if ($fake) {
            $text = "کانال جعلی:\nعنوان: {$fake['title']}\nلینک: {$fake['link']}";
            $keyboard = fake_channel_detail_keyboard($fake_id);
            edit_message_text($chat_id, $message_id, $text, $keyboard);
        } else {
            answer_callback_query($callback_id, 'کانال جعلی یافت نشد.', true);
        }
    }

    // Handle fake channel removal
    elseif (strpos($data, 'fakechannel_remove_') === 0) {
        $fake_id = (int)substr($data, 19);
        remove_fake_channel($fake_id);
        answer_callback_query($callback_id, 'کانال جعلی حذف شد.', true);
        edit_message_text($chat_id, $message_id, '🎭 مدیریت کانال‌های جعلی:', fake_channel_keyboard());
    }

    // Handle fake channel add
    elseif ($data === 'fakechannel_add') {
        set_admin_state($user_id, ['mode' => 'add_fake_channel', 'step' => 'title']);
        edit_message_text(
            $chat_id,
            $message_id,
            "لطفاً عنوان کانال جعلی را ارسال کنید.\nمثال: کانال تبلیغات"
        );
    }

    answer_callback_query($callback_id);
}

/**
 * Handle seen/reaction lock callbacks
 */
function handle_seenlock_callback(array $callback_query): void {
    $callback_id = $callback_query['id'];
    $user_id = $callback_query['from']['id'];
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $data = $callback_query['data'];

    // Check admin permission
    if (!is_admin($user_id)) {
        answer_callback_query($callback_id, 'شما اجازه دسترسی به این بخش را ندارید.', true);
        return;
    }

    // Handle lock info display
    if (strpos($data, 'seenlock_info_') === 0) {
        $lock_id = (int)substr($data, 14);
        $lock = get_seen_reaction_lock_by_id($lock_id);

        if ($lock) {
            $type_fa = $lock['lock_type'] === 'seen' ? 'سین' : 'ریاکشن';
            $text = "قفل {$type_fa}:\n";
            $text .= "ترتیب: #{$lock['lock_order']}\n";
            $text .= "مدت زمان: {$lock['timer_seconds']} ثانیه\n";
            $text .= "متن: {$lock['message_text']}";

            $keyboard = seen_lock_detail_keyboard($lock_id);
            edit_message_text($chat_id, $message_id, $text, $keyboard);
        } else {
            answer_callback_query($callback_id, 'قفل یافت نشد.', true);
        }
    }

    // Handle lock removal
    elseif (strpos($data, 'seenlock_remove_') === 0) {
        $lock_id = (int)substr($data, 16);
        remove_seen_reaction_lock($lock_id);
        answer_callback_query($callback_id, 'قفل حذف شد.', true);
        edit_message_text($chat_id, $message_id, '👁 مدیریت قفل سین/ریاکشن:', seen_lock_keyboard());
    }

    // Handle lock add
    elseif ($data === 'seenlock_add') {
        $lock_count = get_seen_reaction_lock_count();
        if ($lock_count >= 10) {
            answer_callback_query($callback_id, 'حداکثر 10 قفل مجاز است.', true);
            return;
        }

        edit_message_text($chat_id, $message_id, 'نوع قفل را انتخاب کنید:', lock_type_keyboard());
    }

    // Handle lock type selection
    elseif (strpos($data, 'locktype_') === 0) {
        $lock_type = substr($data, 9); // 'seen' or 'reaction'
        set_admin_state($user_id, ['mode' => 'add_seenlock', 'step' => 'message', 'lock_type' => $lock_type]);

        $type_fa = $lock_type === 'seen' ? 'سین' : 'ریاکشن';
        edit_message_text(
            $chat_id,
            $message_id,
            "افزودن قفل {$type_fa}\n\nمتن مورد نظر را ارسال کنید.\n(می‌توانید لینک هم در متن قرار دهید)"
        );
    }

    answer_callback_query($callback_id);
}

/**
 * Handle user lock completion callbacks
 */
function handle_user_lock_callback(array $callback_query): void {
    $callback_id = $callback_query['id'];
    $user_id = $callback_query['from']['id'];
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $data = $callback_query['data'];

    // Handle lock wait (check timer and update button if ready)
    if (strpos($data, 'lock_wait:') === 0) {
        $parts = explode(':', $data);
        $link_code = $parts[1];
        $lock_id = (int)$parts[2];

        // Check if timer has elapsed
        $elapsed = get_user_lock_elapsed_time($user_id, $link_code, $lock_id);
        $lock = get_seen_reaction_lock_by_id($lock_id);

        if (!$lock) {
            answer_callback_query($callback_id, 'قفل یافت نشد.', true);
            return;
        }

        if ($elapsed >= $lock['timer_seconds']) {
            // Timer complete - mark as completed and proceed
            complete_user_lock($user_id, $link_code, $lock_id);

            // Delete the lock message
            telegram_request('deleteMessage', [
                'chat_id' => $chat_id,
                'message_id' => $message_id
            ]);

            // Check for next lock
            $next_lock = get_next_user_lock($user_id, $link_code);

            if ($next_lock) {
                // Show next lock
                start_user_lock($user_id, $link_code, $next_lock['id']);

                $type_fa = $next_lock['lock_type'] === 'seen' ? 'سین' : 'ریاکشن';
                send_message(
                    $chat_id,
                    "قفل {$type_fa} #{$next_lock['lock_order']}:\n\n{$next_lock['message_text']}",
                    user_lock_keyboard($link_code, $next_lock['id'], false)
                );
                answer_callback_query($callback_id, 'قفل بعدی نمایش داده شد.');
            } else {
                // All locks completed - send files
                $file_record = get_file_record($link_code);

                if ($file_record) {
                    send_files_to_user($chat_id, $user_id, $link_code, $file_record);
                    clear_user_lock_progress($user_id, $link_code);
                    answer_callback_query($callback_id, 'فایل ارسال شد!');
                } else {
                    answer_callback_query($callback_id, 'لینک نامعتبر است.', true);
                }
            }
        } else {
            // Timer not complete yet
            $remaining = $lock['timer_seconds'] - $elapsed;
            answer_callback_query($callback_id, "تایید نشد لطفا ۵ ثانیه دیگر تلاش کنید", true);
        }
        return;
    }

    // Handle lock completion
    if (strpos($data, 'lock_complete:') === 0) {
        $parts = explode(':', $data);
        $link_code = $parts[1];
        $lock_id = (int)$parts[2];

        // Check timer
        $elapsed = get_user_lock_elapsed_time($user_id, $link_code, $lock_id);
        $lock = get_seen_reaction_lock_by_id($lock_id);

        if (!$lock) {
            answer_callback_query($callback_id, 'قفل یافت نشد.', true);
            return;
        }

        if ($elapsed < $lock['timer_seconds']) {
            answer_callback_query($callback_id, 'لطفاً صبر کنید...', true);
            return;
        }

        // Mark lock as completed
        complete_user_lock($user_id, $link_code, $lock_id);

        // Delete the lock message
        telegram_request('deleteMessage', [
            'chat_id' => $chat_id,
            'message_id' => $message_id
        ]);

        // Check for next lock
        $next_lock = get_next_user_lock($user_id, $link_code);

        if ($next_lock) {
            // Show next lock
            start_user_lock($user_id, $link_code, $next_lock['id']);

            $type_fa = $next_lock['lock_type'] === 'seen' ? 'سین' : 'ریاکشن';
            send_message(
                $chat_id,
                "قفل {$type_fa} #{$next_lock['lock_order']}:\n\n{$next_lock['message_text']}",
                user_lock_keyboard($link_code, $next_lock['id'], false)
            );

            answer_callback_query($callback_id, 'قفل بعدی نمایش داده شد.');
        } else {
            // All locks completed - send files
            $file_record = get_file_record($link_code);

            if ($file_record) {
                send_files_to_user($chat_id, $user_id, $link_code, $file_record);
                clear_user_lock_progress($user_id, $link_code);
                answer_callback_query($callback_id, 'فایل ارسال شد!');
            } else {
                answer_callback_query($callback_id, 'لینک نامعتبر است.', true);
            }
        }
        return;
    }

    answer_callback_query($callback_id);
}

/**
 * Handle broadcast management callbacks
 */
function handle_broadcast_callback(array $callback_query): void {
    $callback_id = $callback_query['id'];
    $user_id = $callback_query['from']['id'];
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $data = $callback_query['data'];

    // Check admin permission
    if (!is_admin($user_id)) {
        answer_callback_query($callback_id, 'شما اجازه دسترسی به این بخش را ندارید.', true);
        return;
    }

    // Handle broadcast info display
    if (strpos($data, 'broadcast_info_') === 0) {
        $broadcast_id = (int)substr($data, 15);
        $broadcast = get_broadcast_details($broadcast_id);

        if ($broadcast) {
            $text = format_broadcast_details($broadcast);
            $keyboard = broadcast_detail_keyboard($broadcast_id, $broadcast['status']);
            edit_message_text($chat_id, $message_id, $text, $keyboard);
            answer_callback_query($callback_id, 'جزئیات پخش به‌روزرسانی شد.');
        } else {
            answer_callback_query($callback_id, 'پخش یافت نشد.', true);
        }
    }

    // Handle broadcast cancellation
    elseif (strpos($data, 'broadcast_cancel_') === 0) {
        $broadcast_id = (int)substr($data, 17);

        if (cancel_broadcast($broadcast_id)) {
            $broadcast = get_broadcast_details($broadcast_id);
            $text = format_broadcast_details($broadcast);
            $keyboard = broadcast_detail_keyboard($broadcast_id, $broadcast['status']);
            edit_message_text($chat_id, $message_id, $text, $keyboard);
            answer_callback_query($callback_id, '✅ پخش با موفقیت لغو شد.', true);
        } else {
            answer_callback_query($callback_id, '❌ امکان لغو این پخش وجود ندارد.', true);
        }
    }

    // Handle broadcast list pagination
    elseif (strpos($data, 'broadcast_page_') === 0) {
        $page = (int)substr($data, 15);
        edit_message_text($chat_id, $message_id, '📡 مدیریت پخش‌های همگانی:', broadcast_manager_keyboard($page));
        answer_callback_query($callback_id);
    }
}

/**
 * Handle link management callbacks
 */
function handle_link_callback(array $callback_query): void {
    $callback_id = $callback_query['id'];
    $user_id = $callback_query['from']['id'];
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $data = $callback_query['data'];

    // Check admin permission
    if (!is_admin($user_id)) {
        answer_callback_query($callback_id, 'شما اجازه دسترسی به این بخش را ندارید.', true);
        return;
    }

    // Handle link info display
    if (strpos($data, 'link_info_') === 0) {
        $link_code = substr($data, 10);
        $file_record = get_file_record($link_code);

        if ($file_record) {
            $text = format_link_details($link_code, $file_record);
            $keyboard = link_detail_keyboard($link_code);
            edit_message_text($chat_id, $message_id, $text, $keyboard);
            answer_callback_query($callback_id);
        } else {
            answer_callback_query($callback_id, 'لینک یافت نشد.', true);
        }
    }

    // Handle link list pagination
    elseif (strpos($data, 'link_page_') === 0) {
        $page = (int)substr($data, 10);
        edit_message_text($chat_id, $message_id, '🔗 مدیریت لینک‌های صادر شده:', link_manager_keyboard($page));
        answer_callback_query($callback_id);
    }

    // Handle add file to link
    elseif (strpos($data, 'link_addfile_') === 0) {
        $link_code = substr($data, 13);

        set_admin_state($user_id, ['mode' => 'link_addfile', 'link_code' => $link_code]);
        edit_message_text(
            $chat_id,
            $message_id,
            "فایل‌های جدید را برای لینک <code>{$link_code}</code> ارسال کنید.\nپس از اتمام، دستور /done را بفرستید."
        );
        answer_callback_query($callback_id);
    }

    // Handle delete file from link
    elseif (strpos($data, 'link_delfile_') === 0) {
        $link_code = substr($data, 13);
        $file_record = get_file_record($link_code);

        if ($file_record && !empty($file_record['file_data'])) {
            edit_message_text(
                $chat_id,
                $message_id,
                "کدام فایل را از لینک <code>{$link_code}</code> حذف کنیم؟",
                link_file_selection_keyboard($link_code, $file_record['file_data'])
            );
            answer_callback_query($callback_id);
        } else {
            answer_callback_query($callback_id, 'فایلی برای حذف وجود ندارد.', true);
        }
    }

    // Handle remove specific file
    elseif (strpos($data, 'link_removefile_') === 0) {
        $parts = explode('_', $data);
        $link_code = $parts[2];
        $file_index = (int)$parts[3];

        $file_record = get_file_record($link_code);

        if ($file_record && isset($file_record['file_data'][$file_index])) {
            // Remove the file at index
            $new_file_data = $file_record['file_data'];
            array_splice($new_file_data, $file_index, 1);

            // Check if there are any files left
            if (empty($new_file_data)) {
                // Delete the entire link if no files remain
                delete_file_link($link_code);
                answer_callback_query($callback_id, 'لینک حذف شد (تمام فایل‌ها حذف شدند).', true);
                edit_message_text($chat_id, $message_id, '🔗 مدیریت لینک‌های صادر شده:', link_manager_keyboard(0));
            } else {
                // Update with remaining files
                update_file_data($link_code, $new_file_data);
                answer_callback_query($callback_id, 'فایل حذف شد.', true);

                // Refresh link details
                $updated_record = get_file_record($link_code);
                $text = format_link_details($link_code, $updated_record);
                $keyboard = link_detail_keyboard($link_code);
                edit_message_text($chat_id, $message_id, $text, $keyboard);
            }
        } else {
            answer_callback_query($callback_id, 'فایل یافت نشد.', true);
        }
    }

    else {
        answer_callback_query($callback_id);
    }
}

/**
 * Handle chat_member updates (track left users)
 */
function handle_chat_member_update(array $chat_member): void {
    $user_id = $chat_member['from']['id'];
    $chat_id = $chat_member['chat']['id'];
    $old_status = $chat_member['old_chat_member']['status'] ?? null;
    $new_status = $chat_member['new_chat_member']['status'];

    // Check if user left a channel that is in our channel locks
    $channels = get_channels(false); // Get all channels including inactive
    $is_tracked_channel = false;

    foreach ($channels as $channel) {
        if ($channel['chat_id'] == $chat_id) {
            $is_tracked_channel = true;
            break;
        }
    }

    if (!$is_tracked_channel) {
        return;
    }

    // Track if user left (was member or restricted, now left or kicked)
    if (in_array($old_status, ['member', 'restricted', 'administrator', 'creator']) &&
        in_array($new_status, ['left', 'kicked'])) {
        track_channel_interaction($user_id, $chat_id, 'left');
    }
}

/**
 * Handle channel management callbacks (updated with limits)
 */
function handle_channel_callback_extended(array $callback_query): void {
    $callback_id = $callback_query['id'];
    $user_id = $callback_query['from']['id'];
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $data = $callback_query['data'];

    // Check admin permission
    if (!is_admin($user_id)) {
        answer_callback_query($callback_id, 'شما اجازه دسترسی به این بخش را ندارید.', true);
        return;
    }

    // Handle set limits
    if (strpos($data, 'channel_limits_') === 0) {
        $channel_chat_id = (int)substr($data, 15);

        set_admin_state($user_id, [
            'mode' => 'set_channel_limits',
            'chat_id' => $channel_chat_id,
            'step' => 'join_limit'
        ]);

        edit_message_text(
            $chat_id,
            $message_id,
            "⚙️ تنظیم محدودیت کانال\n\n👥 محدودیت تعداد عضو را وارد کنید:\n(برای نامحدود عدد 0 وارد کنید)"
        );
        answer_callback_query($callback_id);
    }
}

/**
 * Handle admin management callbacks
 */
function handle_adminmgr_callback(array $callback_query): void {
   // devLog("ok?");
    $callback_id = $callback_query['id'];
    $user_id = $callback_query['from']['id'];
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $data = $callback_query['data'];

    // Check admin permission
    if (!is_admin($user_id)) {
        answer_callback_query($callback_id, 'شما اجازه دسترسی به این بخش را ندارید.', true);
        return;
    }

    // Handle admin list
    if ($data === 'admin_admins') {
        edit_message_text($chat_id, $message_id, '👥 مدیریت ادمین‌ها:', admin_management_keyboard());
        answer_callback_query($callback_id);
    }

    // Handle admin info
    elseif (strpos($data, 'adminmgr_info_') === 0) {
        $admin_user_id = (int)substr($data, 14);

        $is_config = is_config_admin($admin_user_id);
        $type = $is_config ? 'ادمین اصلی' : 'ادمین اضافه شده';

        $text = "👤 <b>اطلاعات ادمین</b>\n\n";
        $text .= "🆔 User ID: <code>{$admin_user_id}</code>\n";
        $text .= "📋 نوع: {$type}\n";

        if (!$is_config) {
            $admins = get_dynamic_admins();
            foreach ($admins as $admin) {
                if ($admin['user_id'] == $admin_user_id) {
                    $text .= "➕ توسط: {$admin['added_by']}\n";
                    $text .= "📅 تاریخ: " . gregorian_to_jalali($admin['added_at']) . "\n";
                    break;
                }
            }
        }

        edit_message_text($chat_id, $message_id, $text, admin_detail_keyboard($admin_user_id));
        answer_callback_query($callback_id);
    }

    // Handle add admin
    elseif ($data === 'adminmgr_add') {
        set_admin_state($user_id, ['mode' => 'add_admin']);
        edit_message_text(
            $chat_id,
            $message_id,
            "➕ افزودن ادمین جدید\n\nلطفاً User ID ادمین جدید را ارسال کنید:\n(برای لغو /cancel را بزنید)"
        );
        answer_callback_query($callback_id);
    }

    // Handle remove admin
    elseif (strpos($data, 'adminmgr_remove_') === 0) {
        $admin_user_id = (int)substr($data, 16);

        if (remove_dynamic_admin($admin_user_id)) {
            answer_callback_query($callback_id, '✅ ادمین حذف شد.', true);
            edit_message_text($chat_id, $message_id, '👥 مدیریت ادمین‌ها:', admin_management_keyboard());
        } else {
            answer_callback_query($callback_id, '❌ خطا در حذف ادمین.', true);
        }
    }

    else {
        answer_callback_query($callback_id);
    }
}

?>

/**
 * Handle admin subscription callbacks
 */
function handle_admin_subscription_callback(array $callback_query): void {
    $callback_id = $callback_query['id'];
    $user_id = $callback_query['from']['id'];
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $data = $callback_query['data'];

    // Check admin permission
    if (!is_admin($user_id)) {
        answer_callback_query($callback_id, 'شما اجازه دسترسی به این بخش را ندارید.', true);
        return;
    }

    if ($data === 'admin_subscription') {
        edit_message_text($chat_id, $message_id, '💳 مدیریت قفل اشتراک:', admin_subscription_panel_keyboard());
    }

    elseif ($data === 'sub_enable') {
        set_bot_setting('subscription_lock_enabled', '1');
        edit_message_text($chat_id, $message_id, '💳 مدیریت قفل اشتراک:', admin_subscription_panel_keyboard());
        answer_callback_query($callback_id, 'قفل اشتراک فعال شد.');
    }

    elseif ($data === 'sub_disable') {
        set_bot_setting('subscription_lock_enabled', '0');
        edit_message_text($chat_id, $message_id, '💳 مدیریت قفل اشتراک:', admin_subscription_panel_keyboard());
        answer_callback_query($callback_id, 'قفل اشتراک غیرفعال شد.');
    }

    elseif ($data === 'sub_plans') {
        edit_message_text($chat_id, $message_id, '📋 لیست پلن‌ها (برای حذف کلیک کنید):', admin_plans_keyboard());
    }

    elseif ($data === 'sub_add_plan') {
        set_admin_state($user_id, ['mode' => 'sub_add_plan', 'step' => 'name']);
        edit_message_text($chat_id, $message_id, "➕ افزودن پلن جدید\n\nلطفاً نام پلن را ارسال کنید:\n(مثال: اشتراک یک ماهه)");
    }

    elseif (strpos($data, 'sub_del_plan_') === 0) {
        $plan_id = (int)substr($data, 13);
        delete_subscription_plan($plan_id);
        edit_message_text($chat_id, $message_id, '📋 لیست پلن‌ها (برای حذف کلیک کنید):', admin_plans_keyboard());
        answer_callback_query($callback_id, 'پلن حذف شد.');
    }

    elseif ($data === 'sub_bank_info') {
        set_admin_state($user_id, ['mode' => 'sub_bank_info']);
        $current_info = get_bot_setting('bank_info', 'تنظیم نشده');
        edit_message_text($chat_id, $message_id, "💳 تنظیم اطلاعات بانکی\n\nاطلاعات فعلی:\n{$current_info}\n\nلطفاً شماره کارت و نام صاحب حساب را ارسال کنید:");
    }

    elseif ($data === 'sub_set_channel') {
        set_admin_state($user_id, ['mode' => 'sub_set_channel']);
        $current_channel = get_bot_setting('payment_channel_id', 'تنظیم نشده');
        edit_message_text($chat_id, $message_id, "📢 تنظیم کانال واریزی\n\nکانال فعلی: {$current_channel}\n\nلطفاً Chat ID کانال/گروه مورد نظر برای دریافت رسیدها را ارسال کنید:");
    }

    answer_callback_query($callback_id);
}

/**
 * Handle admin subscription messages
 */
function handle_admin_subscription_message(array $message, array $state): void {
    $user_id = $message['from']['id'];
    $chat_id = $message['chat']['id'];
    $text = $message['text'] ?? '';
    $mode = $state['mode'];

    if ($mode === 'sub_add_plan') {
        $step = $state['step'];

        if ($step === 'name') {
            $state['name'] = $text;
            $state['step'] = 'price';
            set_admin_state($user_id, $state);
            send_message($chat_id, "قیمت پلن را به تومان وارد کنید:\n(مثال: 50000)");
        }
        elseif ($step === 'price') {
            if (is_numeric($text)) {
                $state['price'] = (float)$text;
                $state['step'] = 'duration';
                set_admin_state($user_id, $state);
                send_message($chat_id, "مدت زمان پلن را به روز وارد کنید:\n(مثال: 30)");
            } else {
                send_message($chat_id, "لطفاً یک عدد معتبر وارد کنید.");
            }
        }
        elseif ($step === 'duration') {
            if (is_numeric($text)) {
                $duration = (int)$text;
                add_subscription_plan($state['name'], $state['price'], $duration, '');
                clear_admin_state($user_id);
                send_message($chat_id, "✅ پلن با موفقیت ایجاد شد.", admin_plans_keyboard());
            } else {
                send_message($chat_id, "لطفاً یک عدد معتبر وارد کنید.");
            }
        }
    }

    elseif ($mode === 'sub_bank_info') {
        set_bot_setting('bank_info', $text);
        clear_admin_state($user_id);
        send_message($chat_id, "✅ اطلاعات بانکی ذخیره شد.", admin_subscription_panel_keyboard());
    }

    elseif ($mode === 'sub_set_channel') {
        if (is_numeric($text)) {
            set_bot_setting('payment_channel_id', $text);
            clear_admin_state($user_id);
            send_message($chat_id, "✅ کانال واریزی تنظیم شد.", admin_subscription_panel_keyboard());
        } else {
            send_message($chat_id, "لطفاً یک Chat ID معتبر (عدد) وارد کنید.");
        }
    }
}

/**
 * Handle payment callback
 */
function handle_payment_callback(array $callback_query): void {
    $callback_id = $callback_query['id'];
    $user_id = $callback_query['from']['id']; // Admin who clicked
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $data = $callback_query['data'];

    // Extract request ID
    $parts = explode('_', $data);
    $action = $parts[1]; // approve or reject
    $request_id = (int)$parts[2];

    $request = get_payment_request($request_id);
    if (!$request) {
        answer_callback_query($callback_id, 'درخواست یافت نشد.', true);
        return;
    }

    if ($request['status'] !== 'pending') {
        answer_callback_query($callback_id, 'این درخواست قبلاً بررسی شده است.', true);
        return;
    }

    if ($action === 'approve') {
        $plan = get_subscription_plan($request['plan_id']);
        if ($plan) {
            add_user_subscription($request['user_id'], $request['plan_id'], $plan['duration_days']);
            update_payment_request_status($request_id, 'approved');
            
            // Notify user
            send_message($request['user_id'], "✅ تبریک! اشتراک شما فعال شد.\nاکنون می‌توانید از ربات استفاده کنید.");
            
            // Update admin message
            $new_caption = $callback_query['message']['caption'] . "\n\n✅ تایید شد توسط: {$user_id}";
            edit_message_caption($chat_id, $message_id, $new_caption);
            answer_callback_query($callback_id, 'تایید شد.');
        } else {
            answer_callback_query($callback_id, 'خطا: پلن حذف شده است.', true);
        }
    } elseif ($action === 'reject') {
        update_payment_request_status($request_id, 'rejected');
        
        // Notify user
        send_message($request['user_id'], "❌ متاسفانه درخواست اشتراک شما رد شد.\nلطفاً در صورت نیاز با پشتیبانی تماس بگیرید.");
        
        // Update admin message
        $new_caption = $callback_query['message']['caption'] . "\n\n❌ رد شد توسط: {$user_id}";
        edit_message_caption($chat_id, $message_id, $new_caption);
        answer_callback_query($callback_id, 'رد شد.');
    }
}

/**
 * Edit message caption (helper)
 */
function edit_message_caption(int $chat_id, int $message_id, string $caption): void {
    $data = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'caption' => $caption
    ];
    telegram_request('editMessageCaption', $data);
}
?>


